-- $Source: /home/harp/1/proto/monoBANK/xbind/basicwin.adb,v $ 
-- $Revision: 1.12 $ $Date: 96/03/04 12:19:34 $ $Author: mg $ 

-- --------------------------------------------------------------------------
-- THIS FILE AND ANY ASSOCIATED DOCUMENTATION IS FURNISHED "AS IS" WITHOUT 
-- WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED 
-- TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
-- PARTICULAR PURPOSE.  The user assumes the entire risk as to the accuracy 
-- and the use of this file. 
--  
-- Copyright (c) Intermetrics, Inc. 1994 
-- Royalty-free, unlimited, worldwide, non-exclusive use, modification, 
-- reproduction and further distribution of this file is permitted. 
-- --------------------------------------------------------------------------
-- C version Copyright 1989 O'Reilly and Associates, Inc.

-- Translated to Ada by Mitch Gart, 3/27/95
-- From chapter 3, Xlib Programming Manual.

with Ada.Command_Line;
with Ada.Unchecked_Conversion;
with Interfaces.C;
with Interfaces.C.Strings;
with System;
with Text_IO;
with X;
with X.Args;
with X.Strings;
with X.Xlib;
with X.Xutil;

procedure BasicWin is

    BITMAPDEPTH: constant := 1;
    TOO_SMALL  : constant := 0;
    BIG_ENOUGH : constant := 1;

    subtype charp is X.Strings.charp;
    Nul: Interfaces.C.Char renames Interfaces.C.Nul;
    subtype Char_Array is Interfaces.C.Char_Array;
    use type Interfaces.C.Char_Array;
    use type Interfaces.C.Int;
    use type Interfaces.C.Unsigned;
    subtype Int is Interfaces.C.Int;

    -- These are used as arguments to nearly every Xlib routine, so it saves 
    -- routine arguments to declare them global.  If there were 
    -- additional source files, they would be declared extern there. 
    display   : X.Xlib.XDisplay_access;
    screen_num: Interfaces.C.Int;

    progname: constant String := Ada.Command_Line.Command_Name;
					 -- name this program was invoked by 

    procedure exit_prog(n: interfaces.c.int);
    pragma Import(C, exit_prog, "exit");

    dash_list: Interfaces.C.Char_Array(0..1) := (
	Interfaces.C.Char'Val(12), Interfaces.C.Char'Val(24));

    procedure Msg(S: String) is
    begin
	Text_IO.Put_Line(Text_IO.Standard_Error, S);
    end Msg;

    procedure getGC(win      : X.Drawable; 
		    gc       : access X.Xlib.GC; 
		    font_info: access X.Xlib.XFontStruct) is

	valuemask  : Interfaces.C.unsigned_long := 0; 
				       -- ignore XGCvalues and use defaults 
	values     : aliased X.Xlib.XGCValues;
	line_width : Interfaces.C.unsigned := 6;
	line_style : int:= X.LineOnOffDash;
	cap_style  : int:= X.CapRound;
	join_style : int:= X.JoinRound;
	dash_offset: int:= 0;
	list_length: int:= 2;

    begin

	-- Create default Graphics Context 
	gc.all := X.Xlib.XCreateGC(display, win, valuemask, values'access);

	-- specify font 
	X.Xlib.XSetFont(display, gc.all, font_info.fid);

	-- specify black foreground since default window background is 
	-- white and default foreground is undefined. 
	X.Xlib.XSetForeground(display, gc.all, 
		       X.Xlib.BlackPixel(display, screen_num));

	-- set line attributes 
	X.Xlib.XSetLineAttributes(display, gc.all, 
		line_width, line_style, cap_style, join_style);

	-- set dashes 
	X.Xlib.XSetDashes(display, gc.all, 
	    -- dash_offset, dash_list(0)'access, list_length);
	    dash_offset, dash_list, list_length);
    end getGC;

    procedure load_font(font_info: access X.Xlib.XFontStruct_access) is
	use type X.Xlib.XFontStruct_access;
    begin
	-- Load font and get font information structure. 
	font_info.all := X.Xlib.XLoadQueryFont(display, "9x15");
	if font_info.all = null then
	    Msg(progname & ": Cannot open 9x15 font");
	    exit_prog( -1 );
	end if;
    end load_font;

    procedure TooSmall(win      : X.Drawable;
		       gc       : X.Xlib.GC; 
		       font_info: access X.Xlib.XFontStruct) is

	Too_Small: constant Char_Array := "Too Small";
	y_offset, 
	x_offset : Interfaces.C.int;

    begin

	y_offset := font_info.ascent + 2;
	x_offset := 2;

	-- output text, centered on each line 
	X.Xlib.XDrawString(display, win, gc, x_offset, y_offset, 
			   Too_Small, 
		           Too_Small'Length);
    end TooSmall;

    procedure draw_text(win       : X.Drawable;
			gc        : X.Xlib.GC; 
			font_info : access X.Xlib.XFontStruct; 
			win_width,
			win_height: Interfaces.C.unsigned) is

	string1: constant Char_Array := "Hi! I'm a window, who are you?";
	string2: constant Char_Array := "To terminate program; Press any key"; 
	string3: constant Char_Array := "or button while in this window."; 
	string4: constant Char_Array := "Screen Dimensions:"; 
	len1, len2, len3, len4: Interfaces.C.Int;
	width1, width2, width3: Interfaces.C.Int;
	cd_height, cd_width, cd_depth: Interfaces.C.Char_Array(0..49);
	font_height : Interfaces.C.int;
	initial_y_offset, x_offset: Interfaces.C.int;

	procedure sprintf(result: out Interfaces.C.Char_Array;
			  format: String;
			  arg   : Interfaces.C.Int);
	pragma Import(C, sprintf, "sprintf");

    begin


	-- need length for both XTextWidth and XDrawString 
	len1 := string1'Length;
	len2 := string2'Length;
	len3 := string3'Length;

	-- get string widths for centering 
	width1 := X.Xlib.XTextWidth(font_info.all'access, string1, len1);
	width2 := X.Xlib.XTextWidth(font_info.all'access, string2, len2);
	width3 := X.Xlib.XTextWidth(font_info.all'access, string3, len3);

	font_height := font_info.ascent + font_info.descent;

	-- output text, centered on each line 
	X.Xlib.XDrawString(display, win, gc, 
		(Interfaces.C.int(win_width) - width1)/2, 
		font_height,
		string1, len1);
	X.Xlib.XDrawString(display, win, gc, 
		(Interfaces.C.int(win_width) - width2)/2, 
		Interfaces.C.int(win_height) - (2 * font_height),
		string2, len2);
	X.Xlib.XDrawString(display, win, gc, 
		(Interfaces.C.int(win_width) - width3)/2, 
		Interfaces.C.int(win_height) - font_height,
		string3, len3);

	-- copy numbers into string variables 
	sprintf(cd_height, " Height - %d pixels", 
		X.Xlib.DisplayHeight(display, screen_num));
	sprintf(cd_width, " Width  - %d pixels", 
		X.Xlib.DisplayWidth(display, screen_num));
	sprintf(cd_depth, " Depth  - %d plane(s)", 
		X.Xlib.DefaultDepth(display, screen_num));

	-- reuse these for same purpose 
	len1 := Interfaces.C.Int(X.Strings.Strlen(cd_height));
	len2 := Interfaces.C.Int(X.Strings.Strlen(cd_width));
	len3 := Interfaces.C.Int(X.Strings.Strlen(cd_depth));
	len4 := Interfaces.C.Int(X.Strings.Strlen(string4));

	-- To center strings vertically, we place the first string
	-- so that the top of it is two font_heights above the center
	-- of the window.  Since the baseline of the string is what we
	-- need to locate for XDrawString, and the baseline is one
	-- font_info.ascent below the top of the character,
	-- the final offset of the origin up from the center of the 
	-- window is one font_height + one descent. 

	initial_y_offset := Interfaces.C.Int(win_height/2) - font_height - 
			    Interfaces.C.Int(font_info.descent);
	x_offset := Interfaces.C.int(win_width/4);
	X.Xlib.XDrawString(display, win, gc, x_offset, 
		    Interfaces.C.int(initial_y_offset),
		    string4, len4);
	X.Xlib.XDrawString(display, win, gc, x_offset, 
		    Interfaces.C.int(initial_y_offset) + 
		    font_height, cd_height, len1);
	X.Xlib.XDrawString(display, win, gc, x_offset, 
		    Interfaces.C.int(initial_y_offset) + 
		    2 * font_height, cd_width, len2);
	X.Xlib.XDrawString(display, win, gc, x_offset, 
		    Interfaces.C.int(initial_y_offset) + 
		    3 * font_height, cd_depth, len3);
    end draw_text;

    procedure draw_graphics(win          : X.Drawable;
			    gc           : X.Xlib.GC; 
			    window_width,
			    window_height: Interfaces.C.unsigned) is
	xx, 
	yy    : Interfaces.C.int;
	width, 
	height: Interfaces.C.unsigned;
    begin
	height := window_height/2;
	width := 3 * window_width/4;
	xx := Interfaces.C.int(window_width/2 - width/2);  -- center 
	yy := Interfaces.C.int(window_height/2 - height/2);
	X.Xlib.XDrawRectangle(display, win, gc, xx, yy, width, height);
    end draw_graphics;

    -- #include "../bitmaps/icon_bitmap"
    type bitmap is array (natural range <>) of Interfaces.C.Unsigned_Char;
    icon_bitmap_bits: constant bitmap := (
       16#c3#, 16#c3#, 16#7f#, 16#00#, 16#78#, 16#00#, 
       16#00#, 16#00#, 16#00#, 16#c0#, 16#00#, 16#00#,
       16#00#, 16#00#, 16#80#, 16#38#, 16#00#, 16#40#, 
       16#00#, 16#80#, 16#24#, 16#00#, 16#00#, 16#00#,
       16#80#, 16#44#, 16#00#, 16#00#, 16#00#, 16#80#, 
       16#44#, 16#00#, 16#00#, 16#00#, 16#80#, 16#74#,
       16#00#, 16#0f#, 16#0c#, 16#00#, 16#7c#, 16#3e#, 
       16#41#, 16#0e#, 16#00#, 16#44#, 16#22#, 16#41#,
       16#02#, 16#00#, 16#84#, 16#22#, 16#46#, 16#02#, 
       16#00#, 16#9c#, 16#26#, 16#cc#, 16#02#, 16#00#,
       16#78#, 16#3c#, 16#cd#, 16#36#, 16#80#, 16#00#, 
       16#20#, 16#06#, 16#0c#, 16#80#, 16#01#, 16#00#,
       16#00#, 16#00#, 16#80#, 16#01#, 16#02#, 16#40#, 
       16#00#, 16#80#, 16#01#, 16#06#, 16#40#, 16#00#,
       16#80#, 16#01#, 16#04#, 16#20#, 16#00#, 16#80#, 
       16#01#, 16#04#, 16#20#, 16#01#, 16#80#, 16#01#,
       16#04#, 16#20#, 16#00#, 16#80#, 16#01#, 16#04#, 
       16#22#, 16#00#, 16#80#, 16#01#, 16#04#, 16#33#,
       16#f1#, 16#81#, 16#01#, 16#88#, 16#12#, 16#31#, 
       16#03#, 16#01#, 16#88#, 16#12#, 16#11#, 16#02#,
       16#00#, 16#88#, 16#12#, 16#11#, 16#02#, 16#00#, 
       16#48#, 16#1a#, 16#11#, 16#02#, 16#00#, 16#70#,
       16#04#, 16#19#, 16#82#, 16#01#, 16#00#, 16#00#, 
       16#00#, 16#80#, 16#01#, 16#00#, 16#00#, 16#38#,
       16#80#, 16#01#, 16#00#, 16#00#, 16#ce#, 16#80#, 
       16#01#, 16#00#, 16#00#, 16#83#, 16#81#, 16#81#,
       16#07#, 16#80#, 16#01#, 16#81#, 16#e1#, 16#04#, 
       16#c0#, 16#00#, 16#83#, 16#31#, 16#08#, 16#40#,
       16#00#, 16#82#, 16#10#, 16#08#, 16#20#, 16#00#, 
       16#82#, 16#19#, 16#10#, 16#30#, 16#00#, 16#86#,
       16#0c#, 16#30#, 16#18#, 16#00#, 16#84#, 16#04#, 
       16#60#, 16#0e#, 16#00#, 16#dc#, 16#02#, 16#80#,
       16#03#, 16#00#, 16#70#, 16#00#, 16#00#, 16#00#, 16#00#, 16#00#);

    function To_Charp is new Ada.Unchecked_Conversion(
	System.Address, X.Strings.const_charp);

    win                : X.Drawable;
    width, 
    height             : Interfaces.C.Unsigned;    -- window size 
    xx, 
    yy                 : Interfaces.C.Int;     -- window position 
    border_width       : Interfaces.C.Unsigned := 4;    -- four pixels 
    display_width, 
    display_height     : Interfaces.C.Unsigned;
    icon_bitmap_width  : constant := 40; 
    icon_bitmap_height : constant := 40; 
    icon_pixmap        : X.Pixmap;
    size_hints         : aliased X.Xutil.XSizeHints;
    size_list          : aliased X.Xutil.XIconSize_access;
    count              : aliased Interfaces.C.Int;
    report             : aliased X.Xlib.XEvent;
    gc                 : aliased X.Xlib.GC;
    font_info          : aliased X.Xlib.XFontStruct_access;
    window_size        : Interfaces.C.int := BIG_ENOUGH;    
				    -- or TOO_SMALL to display contents 

    wm_hints: aliased X.Xutil.XWMHints;
    class_hints: aliased X.Xutil.XClassHint;

    -- format of the window name and icon name 
    -- arguments has changed in R4
    windowName, 
    iconName    : aliased X.Xutil.XTextProperty;
    BasicWin    : Char_Array := "Basicwin" & Nul;
    window_name : aliased Char_Array := "Basic Window Program" & Nul;
    icon_name   : aliased Char_Array := "basicwin" & Nul;
    wnp         : X.Strings.charp_array(0..0) := 
		      (0 => window_name(0)'unchecked_access);
    inp         : X.Strings.charp_array(0..0) :=
		      (0 => icon_name(0)'unchecked_access);
    use type X.Xlib.XDisplay_access;

begin 		-- BasicWin

    -- connect to X server 
    display := X.Xlib.XOpenDisplay(null);
    if display = null  then
	Msg(progname & ": cannot connect to X server " &
	    Interfaces.C.Strings.Value(X.Strings.To_Chars_Ptr(
			 X.Xlib.XDisplayName(null))));
	exit_prog( -1 );
    end if;

    -- get screen size from display structure macro 
    screen_num     := X.Xlib.DefaultScreen(display);
    display_width  := Interfaces.C.Unsigned(
			  X.Xlib.DisplayWidth(display, screen_num));
    display_height := Interfaces.C.Unsigned(
			  X.Xlib.DisplayHeight(display, screen_num));

    -- Note that in a real application, xx and yy would default to 0
    -- but would be settable from the command line or resource database.  
    --/
    xx := 0;
    yy := 0;

    -- size window with enough room for text 
    width  := display_width/3;
    height := display_height/4;

    -- create opaque window 
    win := X.Drawable(X.Xlib.XCreateSimpleWindow(display, 
	    X.Xlib.RootWindow(display, screen_num), 
	    xx, yy, width, height, border_width, 
	    X.Xlib.BlackPixel(display, screen_num), 
	    X.Xlib.WhitePixel(display, screen_num)));

    -- Get available icon sizes from Window manager 

    if (X.Xutil.XGetIconSizes(display, 
	    X.Xlib.RootWindow(display, screen_num), 
	    size_list'access, count'access) = 0) then
	Msg(": Window manager didn't set icon sizes - using default.");
    else 
	null;
	-- A real application would search through size_list
	-- here to find an acceptable icon size, and then
	-- create a pixmap of that size.  This requires
	-- that the application have data for several sizes
	-- of icons. 
    end if;

    -- Create pixmap of depth 1 (bitmap) for icon
    icon_pixmap := X.Xlib.XCreateBitmapFromData(display, win, 
	    To_Charp(icon_bitmap_bits(0)'Address), 
	    icon_bitmap_width, icon_bitmap_height);

    -- Set size hints for window manager.  The window manager may
    -- override these settings.  Note that in a real
    -- application if size or position were set by the user
    -- the flags would be UPosition and USize, and these would
    -- override the window manager's preferences for this window.
    -- xx, yy, width, and height hints are now taken from
    -- the actual settings of the window when mapped. Note
    -- that PPosition and PSize must be specified anyway.

    size_hints.flags := 
	Interfaces.C.Long(Interfaces.C.Unsigned(
	X.Xutil.PPosition) or X.Xutil.PSize or X.Xutil.PMinSize);
    size_hints.min_width := 300;
    size_hints.min_height := 200;

    -- These calls store window_name and icon_name into
    -- XTextProperty structures and set their other 
    -- fields properly.
    if (X.Xutil.XStringListToTextProperty(wnp(0)'unchecked_access, 1, 
				  windowName'unchecked_access) = 0) then
	Msg(progname & "structure allocation for windowName failed.");
	exit_prog(-1);
    end if;
	
    if (X.Xutil.XStringListToTextProperty(inp(0)'unchecked_access, 1, 
				  iconName'unchecked_access) = 0) then
	Msg(progname & ": structure allocation for iconName failed.");
	exit_prog(-1);
    end if;

    wm_hints.initial_state := X.Xutil.NormalState;
    wm_hints.input := X.Xlib.True;
    wm_hints.icon_pixmap := icon_pixmap;
    wm_hints.flags := Interfaces.C.Long(Interfaces.C.Unsigned(
		      X.Xutil.StateHint) or 
		      X.Xutil.IconPixmapHint or X.Xutil.InputHint);

    class_hints.res_name := X.Strings.Addr(progname);
    class_hints.res_class := Basicwin(0)'unchecked_access;

    X.Xutil.XSetWMProperties(display, X.Window(win), 
	    windowName'access, 
	    iconName'access, 
	    X.Args.argv, X.Args.argc, 
	    size_hints'access, wm_hints'access, 
	    class_hints'access);

    -- Select event types wanted 
    X.Xlib.XSelectInput(display, X.Window(win), 
	    Interfaces.C.Long(Interfaces.C.Unsigned(
		X.ExposureMask) or X.KeyPressMask or 
		X.ButtonPressMask or X.StructureNotifyMask));

    load_font(font_info'access);

    -- create GC for text and drawing 
    getGC(win, gc'access, font_info);

    -- Display window 
    X.Xlib.XMapWindow(display, X.Window(win));

    -- get events, use first to display text and graphics 
    loop
	X.Xlib.XNextEvent(display, report'access);
	case  (report.Event_Type) is
	    when X.Expose =>
		-- unless this is the last contiguous expose,
		-- don't draw the window 
		if (report.xexpose.count = 0) then
		    -- if window too small to use 
		    if (window_size = TOO_SMALL) then
			TooSmall(win, gc, font_info);
		    else 
			-- place text in window 
			draw_text(win, gc, font_info, width, height);

			-- place graphics in window, 
			draw_graphics(win, gc, width, height);
		    end if;
		end if;
	    when X.ConfigureNotify =>
		-- window has been resized, change width and
		-- height to send to draw_text and draw_graphics
		-- in next Expose 
		width := Interfaces.C.Unsigned(report.xconfigure.width);
		height := Interfaces.C.Unsigned(report.xconfigure.height);
		if ((width < Interfaces.C.Unsigned(size_hints.min_width)) or 
		   (height < Interfaces.C.Unsigned(size_hints.min_height))) then
		    window_size := TOO_SMALL;
		else
		    window_size := BIG_ENOUGH;
		end if;
	    when X.ButtonPress | X.KeyPress =>
		X.Xlib.XUnloadFont(display, font_info.fid);
		X.Xlib.XFreeGC(display, gc);
		X.Xlib.XCloseDisplay(display);
		exit_prog(0);
	    when others =>
		-- all events selected by StructureNotifyMask
		-- except ConfigureNotify are thrown away here,
		-- since nothing is done with them 
		null;
	end case;
    end loop;

end BasicWin;
