--------------------------------------------------------------------------------
-- PACKAGE: Pipe Commands
--
-- PURPOSE: Provide a thick Ada binding to the UNX / WIN32 popen and pclose
--          commands. This allows an Ada program to call another program and
--          either send output to that program, or read input from that
--          program.
--
-- USAGE: Execute the command with mode read_file when you want to read the
-- output of the program you start. Execute the command with write_mode when
-- you want to write to the input of the program you start.
--
-- EXCEPTIONS:
--   Access_Error => Raised when a mode violation is attempted
--   End_Of_file  => Raised when the pipe is closed upon a read
--------------------------------------------------------------------------------

   with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;
	with System;

   package Pipe_Commands is
   
      type stream is private;
   
      type IO_MODE is (read_file, write_file);
   
      function execute (Command : in string; IO_type : in IO_Mode)
      return stream;
   
      function read_next (FromFile : in stream)
      return unbounded_string;
   
      procedure write_next (ToFile : in stream; Message : in string);
   
      procedure close (OpenFile : in stream);
   
      Access_Error : exception; -- Raised when attempt is made to violate IO_MODE
      End_Of_File  : exception; -- Raised on detection of End_of_file during read
   
   private
   
      subtype Files is System.Address; -- Corresponds to a C file pointer
   
      type stream is record
            FileStream : Files;
            Mode       : IO_Mode;
         end record;
   
   end Pipe_Commands;
