WITH Ada.Numerics;
WITH Ada.Numerics.Elementary_Functions;
USE  Ada.Numerics;
USE  Ada.Numerics.Elementary_Functions;
WITH Ada.Numerics.Discrete_Random;
WITH Ada.Numerics.Float_Random;
WITH AdaGraph;
PACKAGE BODY Spider_Hires IS
----------------------------------------------------------------------
--| This package provides procedures to emulate "Spider"
--| commands. The spider can move around the screen drawing
--| simple patterns. This version is for a VGA screen.
--| Original Author: John Dalbey, Cal Poly San Luis Obispo, 1992
--| Adapted by: Michael B. Feldman, The George Washington University
--| Last Modified: January 1999
----------------------------------------------------------------------

  -- Types and Objects

  -- Spider's View of her Room - rows and cols both numbered 1..300
  SUBTYPE Rows IS Steps;
  SUBTYPE Cols IS Steps;
  RowsInRoom   : CONSTANT Positive := Rows'Last;
  ColsInRoom   : CONSTANT Positive := Cols'Last;
  -- required by AdaGraph: window and character bounds
  XMax : Positive;
  YMax : Positive;
  XChar : Positive;
  YChar : Positive;

  BackgroundColor : CONSTANT AdaGraph.Color_Type := AdaGraph.Yellow;
  RoomColor       : CONSTANT AdaGraph.Color_Type := AdaGraph.White;
  RowLow       : Positive;  -- room row bounds in window coords
  RowHigh      : Positive;
  ColLow       : Positive;  -- room column bounds in window coords
  ColHigh      : Positive;

  RadiansPerDegree : CONSTANT Float := Pi / 180.0;
                                        -- radians per degree


  -- Spider Current State Information
  CurrentColumn: Cols;                  -- spider's position
  CurrentRow   : Rows;                  --   in the room.
  Heading      : Degrees;               -- spider's direction;
                                        -- North = 0; clockwise
  Ink          : Colors;                -- spider's color
  PlotColor    : AdaGraph.Color_Type := AdaGraph.Black;

  ExecutionSpeed: Speeds;
  DebugFlag: Boolean;


  -- internal procedures and functions, not in specification
  -- and therefore not available to client program

  PROCEDURE DrawRoom  IS
  -- Pre:  None
  -- Post: Room appears on the screen
  BEGIN
    AdaGraph.Draw_Box(X1 => ColLow, X2 => ColHigh,
                      Y1 => RowLow, Y2 => RowHigh,
                      Hue => RoomColor,
                      Filled => AdaGraph.Fill);
    AdaGraph.Goto_XY(X => ColLow, Y => RowHigh);
    AdaGraph.Draw_To(X => ColHigh, Y => RowHigh,
                     Hue => AdaGraph.Black);
    AdaGraph.Draw_To(X => ColHigh, Y => RowLow,
                     Hue => AdaGraph.Black);
    AdaGraph.Draw_To(X => ColLow, Y => RowLow,
                     Hue => AdaGraph.Black);
    AdaGraph.Draw_To(X => ColLow, Y => RowHigh,
                     Hue => AdaGraph.Black);
  END DrawRoom;

  PROCEDURE ChangeColor (NewColor : Colors) IS
  -- Pre:  NewColor is defined
  -- Post: Ink is changed to NewColor and displayed in status box
  BEGIN
    Ink := NewColor;
    CASE Ink IS
      WHEN None =>
        PlotColor := RoomColor;
      WHEN Red =>
        PlotColor := AdaGraph.Red;
      WHEN Blue =>
        PlotColor := AdaGraph.Blue;
      WHEN Green =>
        PlotColor := AdaGraph.Green;
      WHEN Black =>
        PlotColor := AdaGraph.Black;
    END CASE;
  END ChangeColor;

  PROCEDURE ShowSpider IS
  -- Pre:  None
  -- Post: The spider symbol appears in its current position
  BEGIN
    AdaGraph.Put_Pixel(X => ColLow + CurrentColumn,
                       Y => RowHigh - CurrentRow,
                       Hue => PlotColor);
  END ShowSpider;

  -- Random number generators; instances of the generic one
  PACKAGE RandomSteps IS NEW Ada.Numerics.Discrete_Random
    (Result_Subtype => Steps);
  GSteps: RandomSteps.Generator;

  PACKAGE RandomColors IS NEW Ada.Numerics.Discrete_Random
    (Result_Subtype => Colors);
  GColors : RandomColors.Generator;

  PACKAGE RandomDirections IS NEW Ada.Numerics.Discrete_Random
    (Result_Subtype => Directions);
  GDirections : RandomDirections.Generator;

  -- These methods are in the package specification
  -- and implement the "official" spider commands

  PROCEDURE Start IS
  BEGIN
    DrawRoom;
    CurrentColumn := ColsInRoom/2; -- these are in the spider's view
    CurrentRow :=    RowsInRoom/2;
    Heading := 0.0;
    ChangeColor(NewColor => Black);
    ShowSpider;
    RandomSteps.Reset(Gen => GSteps);
    RandomColors.Reset(Gen => GColors);
    RandomDirections.Reset(Gen => GDirections);
  END Start;

  PROCEDURE Face (WhichWay: IN Directions) IS
  BEGIN
    Heading := Float(Directions'Pos(WhichWay) * 90);
  END Face;

  PROCEDURE Step(HowMany: Steps := 1) IS
    NewCol: Integer; -- in room coordinates
    NewRow: Integer; -- in room coordinates
    Dx: Float;
    Dy: Float;
    NearWall: Boolean := False;
  BEGIN
    Dy := Float(HowMany) * Cos(Heading * RadiansPerDegree);
    Dx := Float(HowMany) * Sin(Heading * RadiansPerDegree);
    NewCol := CurrentColumn + Integer(Dx);
    NewRow := CurrentRow + Integer(Dy);
    IF NewCol > Cols'Last THEN      -- clip to room bounds
      NewCol := Cols'Last;
      NearWall := True;
    ELSIF NewCol < Cols'First THEN
      NewCol := Cols'First;
      NearWall := True;
    END IF;
    IF NewRow > Rows'Last THEN
      NewRow := Rows'Last;
      NearWall := True;
    ELSIF NewRow < Rows'First THEN
      NewRow := Rows'First;
      NearWall := True;
    END IF;
    AdaGraph.Draw_Line(X1 => ColLow + CurrentColumn,
                       X2 => ColLow + NewCol,
                       Y1 => RowHigh - CurrentRow,
                       Y2 => RowHigh - NewRow,
                       Hue => PlotColor);
    CurrentRow := NewRow;
    CurrentColumn := NewCol;
       -- if debug mode, wait for mouse click
    IF Debugging = On THEN
      Wait;
    ELSIF ExecutionSpeed = Medium THEN
      DELAY 0.05;
    ELSIF ExecutionSpeed = Slow THEN
      DELAY 0.5;
    END IF;

    IF NearWall THEN
      RAISE Hit_The_Wall;
    ELSE
      RETURN;
    END IF;

  END Step;

  PROCEDURE TurnRight(HowFar: Degrees := 90.0) IS
  BEGIN
    IF Heading + HowFar > 360.0 THEN
      Heading := Heading + HowFar - 360.0;
    ELSE
      Heading := Heading + HowFar;
    END IF;
  END TurnRight;

  FUNCTION IsPainting RETURN Colors IS
  BEGIN
    RETURN Ink;
  END IsPainting;

  FUNCTION IsFacing RETURN Directions IS
  BEGIN
    RETURN Directions'Val(Integer(Heading/90.0));
  END IsFacing;

  FUNCTION AtWall RETURN Boolean IS
  BEGIN
    -- Check for out of bounds (in the spider's view)
    IF Heading < 45.0 THEN
      RETURN CurrentRow >= Rows'Last;
    ELSIF Heading < 135.0 THEN
      RETURN CurrentColumn >= Cols'Last;
    ELSIF Heading < 225.0 THEN
      RETURN CurrentRow <= Rows'First;
    ELSE
      RETURN CurrentColumn <= Cols'First;
    END IF;
  END AtWall;

  FUNCTION RandomStep RETURN Steps IS
  BEGIN
    RETURN RandomSteps.Random(GSteps);
  END RandomStep;

  FUNCTION RandomColor RETURN Colors IS
  BEGIN
    RETURN RandomColors.Random(GColors);
  END RandomColor;

  FUNCTION RandomDirection RETURN Directions IS
  BEGIN
    RETURN RandomDirections.Random(GDirections);
  END RandomDirection;

  GTurn: Ada.Numerics.Float_Random.Generator;
  FUNCTION RandomTurn RETURN Degrees IS
    Turn: Degrees;
  BEGIN
    Turn :=
      Ada.Numerics.Float_Random.Random(GTurn) * Degrees'Last;
    RETURN Turn;
  END RandomTurn;

  PROCEDURE Quit IS
  BEGIN
    Wait;
  END Quit;

  PROCEDURE Debug (Setting: Switch) is
  -- Toggle debugging mode
  BEGIN
    IF Setting = ON THEN
      DebugFlag := true;
    ELSE
      DebugFlag := false;
    END IF;
  END Debug;

  FUNCTION Debugging RETURN Switch IS
  BEGIN
    IF DebugFlag THEN
      RETURN On;
    ELSE
      RETURN Off;
    END IF;
  END Debugging;

  PROCEDURE Wait IS
    USE AdaGraph;
    Mouse: AdaGraph.Mouse_Type;
  BEGIN
    LOOP
      IF AdaGraph.Mouse_Event THEN
        Mouse := AdaGraph.Get_Mouse;
        EXIT WHEN
          Mouse.Event = AdaGraph.Left_Down OR
          Mouse.Event = AdaGraph.Right_Down;
      END IF;
    END LOOP;
  END Wait;

  PROCEDURE SetSpeed (Speed: IN Speeds) IS
  BEGIN
    ExecutionSpeed := Speed;
  END SetSpeed;

BEGIN -- Spider

  -- Initialize everything

  AdaGraph.Create_Graph_Window(XMax, YMax, XChar, YChar);
  AdaGraph.Set_Window_Title(Title => "Spider");
  AdaGraph.Clear_Window(Hue => BackgroundColor);

  RowLow := YMax/2 - RowsInRoom/2;
  RowHigh := RowLow + RowsInRoom;
  ColLow := XMax/2 - ColsInRoom/2;
  ColHigh := ColLow + ColsInRoom;

  SetSpeed (Speed => Fast);

END Spider_Hires;
