-----------------------------------------------------------------------
--
--  File:        demo.adb
--  Description: demonstration of AdaGraph usage
--  Rev:         0.5c
--  Date:        23-jan-1999
--  Author:      Jerry van Dijk
--  Mail:        jdijk@acm.org
--
--  Copyright (c) Jerry van Dijk, 1997, 1998, 1999
--  Billie Hollidaystraat 28
--  2324 LK Leiden
--  THE NETHERLANDS
--  tel int +31 (0)71 531 4365
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------

with Adagraph;    use Adagraph;
with Ada.Text_IO; use Ada.Text_IO;

with Ada.Numerics.Discrete_Random;

procedure Demo is

   ----------------------
   -- Global constants --
   ----------------------
   X_Size : constant Integer := 500;    -- Horizontal window size
   Y_Size : constant Integer := 300;    -- Vertical window size

   ----------------------
   -- Global variables --
   ----------------------
   X_Max,  Y_Max  : Integer;  -- Maximum screen coordinates
   X_Char, Y_Char : Integer;  -- Character size

   -----------------------------------------------------------
   -- Wait for a the spacebar pressed in the command window --
   ------------------------------------------------------------
   procedure Wait_For_Keypress (Msg : in String) is
      C : Character := 'X';
   begin
      Put("Press <SPACEBAR> to " & Msg & ": ");
      while C /= ' ' loop
         Get_Immediate (C);
      end loop;
      New_Line;
   end Wait_For_Keypress;

   --------------------------------
   -- Open graphic output window --
   --------------------------------
   procedure Init_Display is
   begin
      Wait_For_Keypress ("open a 500x300 graphic output window");
      Create_Sized_Graph_Window (X_Size, Y_Size, X_Max, Y_Max, X_Char, Y_Char);
      Set_Window_Title ("AdaGraph v0.5c demo program");
   end Init_Display;

   ---------------------------------
   -- Close graphic output window --
   ---------------------------------
   procedure Kill_Display is
   begin
      Wait_For_Keypress ("close the graphic output window");
      Destroy_Graph_Window;
   end Kill_Display;

   ---------------------------------
   -- Show max size demonstration --
   ---------------------------------
   procedure Show_Max_Size is
      X, Y: Integer;
   begin
      Put_Line ("Window activated: " & Boolean'Image (Is_Open));
      Get_Max_Size (X, Y);
      Put ("Maximum drawing area is 0 .." & Integer'Image (X - 1));
      Put_Line (" by 0 .." & Integer'Image (Y - 1));
      Put ("Current drawing area is 0 .." & Integer'Image (X_Max));
      Put_Line (" by 0 .." & Integer'Image (Y_Max));
   end Show_Max_Size;

   --------------------------------
   -- Clear window demonstration --
   --------------------------------
   procedure Show_Clear_Window is
   begin
      Wait_For_Keypress ("cycle through the available colors");
      for Color in Color_Type'range loop
         Clear_Window (Color);
         delay 1.0;
      end loop;
      Clear_Window;
   end Show_Clear_Window;

   -----------------------------
   -- Put pixel demonstration --
   -----------------------------
   procedure Show_Put_Pixel is

      subtype X_Loc is Positive range 2 .. X_Size - 1;
      subtype Y_Loc is Positive range 2 .. Y_Size - 1;
      subtype The_Color is Color_Type range Blue .. White;

      package X_Random is new Ada.Numerics.Discrete_Random (X_Loc);
      package Y_Random is new Ada.Numerics.Discrete_Random (Y_Loc);
      package C_Random is new Ada.Numerics.Discrete_Random (The_Color);

      Gx : X_Random.Generator;
      Gy : Y_Random.Generator;
      Gc : C_Random.Generator;

   begin
      X_Random.Reset (Gx, 28);
      Y_Random.Reset (Gy, 39);
      C_Random.Reset (Gc, 215);
      Wait_For_Keypress ("display 2000 colored pixels");
      Clear_Window;
      for I in 1 .. 2000 loop
         Put_Pixel (X_Random.Random (Gx),
                    Y_Random.Random (Gy),
                    C_Random.Random (Gc));
      end loop;
   end Show_Put_Pixel;

   -----------------------------
   -- Draw line demonstration --
   -----------------------------
   procedure Show_Draw_Line is
      I    : Integer;
      Step : constant := 5;
   begin
      Wait_For_Keypress ("draw lines");
      Clear_Window;
      I := 0;
      while I < X_Max - Step loop
         Draw_Line(0, 0, I, Y_Max, Light_Blue);
         I := I + Step;
      end loop;
      I := 0;
      while I < X_Max - Step loop
         Draw_Line(X_Max, 0, I, Y_Max, Light_Red);
         I := I + Step;
      end loop;
      I := 0;
      while I < X_Max - Step loop
         Draw_Line(0, Y_Max, I, 0, Light_Cyan);
         I := I + Step;
      end loop;
      I := 0;
      while I < X_Max - Step loop
         Draw_Line(X_Max, Y_Max, I, 0, Light_Magenta);
         I := I + Step;
      end loop;
   end Show_Draw_Line;

   ----------------------------
   -- Draw box demonstration --
   ----------------------------
   procedure Show_Draw_Box is

      procedure Do_It(Filled : in Fill_Type) is
         X1     : Integer    := 0;
         Y1     : Integer    := 0;
         Step   : constant   := 12;
         X2     : Integer    := X_Max;
         Y2     : Integer    := Y_Max;
         Color  : Color_Type := Color_Type'First;
      begin
         Clear_Window;
         while (X2 > X1) and (Y2 > Y1) loop
            Draw_Box (X1, Y1, X2, Y2, Color, Filled);
            X1 := X1 + Step; Y1 := Y1 + Step;
            X2 := X2 - Step; Y2 := Y2 - Step;
            if Color = Color_Type'Last then
               Color := Color_Type'First;
            else
               Color := Color_Type'Succ (Color);
            end if;
         end loop;
      end Do_It;

   begin
      Wait_For_Keypress ("draw filled boxes");
      Do_It (Fill);
      Wait_For_Keypress ("draw unfilled boxes");
      Do_It (No_Fill);
   end Show_Draw_Box;

   -----------------------------
   -- Get pixel demonstration --
   -----------------------------
   procedure Show_Get_Pixel is

      procedure Get_Color (X, Y : in Integer) is
         Color : Color_Type := Get_Pixel (X, Y);
      begin
         Put ("Color at location" & Integer'Image (X));
         Put ("," & Integer'Image (Y) & " is ");
         Put_Line (Color_Type'Image (Color));
      end Get_Color;

   begin
      Wait_For_Keypress ("read pixel colors");
      Clear_Window;
      Draw_Box (10, 10, 20, 20, Light_Red, Fill);
      Get_Color (0, 0);
      Get_Color (12, 12);
   end Show_Get_Pixel;

   -------------------------------
   -- Draw circle demonstration --
   -------------------------------
   procedure Show_Draw_Circle is

      procedure Do_It(Filled : in Fill_Type) is
         Radius : Integer;
         Step   : constant   := 12;
         Color  : Color_Type := Color_Type'First;
         X      : Integer    := X_Max / 2;
         Y      : Integer    := Y_Max / 2;
      begin
         if (X_Max <= Y_Max) then
            Radius := (X_Max / 2) - 2;
         else
            Radius := (Y_Max / 2) - 2;
         end if;
         Clear_Window;
         while Radius > 0 loop
            Draw_Circle (X, Y, Radius, Color, Filled);
            if Color = Color_Type'Last then
               Color := Color_Type'First;
            else
               Color := Color_Type'Succ (Color);
            end if;
            Radius := Radius - Step;
         end loop;
      end Do_It;

   begin
      Wait_For_Keypress ("draw filled circles");
      Do_It (Fill);
      Wait_For_Keypress ("draw unfilled circles");
      Do_It (No_Fill);
   end Show_Draw_Circle;

   --------------------------------
   -- Draw ellipse demonstration --
   --------------------------------
   procedure Show_Draw_Ellipse is

      procedure Do_It(Filled : in Fill_Type) is
         Step  : constant   := 12;
         X1    : Integer    := 0;
         Y1    : Integer    := 0;
         X2    : Integer    := X_Max;
         Y2    : Integer    := Y_Max;
         Color : Color_Type := Color_Type'First;
      begin
         Clear_Window;
         while X1 < X2 and Y1 < Y2 loop
            Draw_Ellipse (X1, Y1, X2, Y2, Color, Filled);
            if Color = Color_Type'Last then
               Color := Color_Type'First;
            else
               Color := Color_Type'Succ (Color);
            end if;
            X1 := X1 + Step;
            Y1 := Y1 + Step;
            X2 := X2 - Step;
            Y2 := Y2 - Step;
         end loop;
      end Do_It;

   begin
      Wait_For_Keypress ("draw filled ellipses");
      Do_It (Fill);
      Wait_For_Keypress ("draw unfilled ellipses");
      Do_It (No_Fill);
   end Show_Draw_Ellipse;

   ------------------------------
   -- Flood fill demonstration --
   ------------------------------
   procedure Show_Flood_Fill is

      procedure Display_Cube is
      begin
         Clear_Window;
         Draw_Box (60, 60, 160, 160);
         Draw_Line (60, 60, 110, 40);
         Draw_Line (160, 60, 210, 40);
         Draw_Line (110, 40, 210, 40);
         Draw_Line (160, 160, 210, 140);
         Draw_Line (210, 40, 210, 140);
      end Display_Cube;

      procedure Fill_Cube is
      begin
         Flood_Fill (61, 61, Light_Blue);
         Flood_Fill (100, 50, Light_Cyan);
         Flood_Fill (161, 61, Blue);
         Flood_Fill (0, 0, Yellow);
      end Fill_Cube;

   begin
      Wait_For_Keypress ("make a drawing");
      Display_Cube;
      Wait_For_Keypress ("floodfill the drawing");
      Fill_Cube;
   end Show_Flood_Fill;

   -----------------------------
   -- Poly draw demonstration --
   -----------------------------
   procedure Show_Poly_Draw is
      X, Y : Integer;
   begin
      Wait_For_Keypress ("show the polydraw support");
      Clear_Window;
      Goto_XY (10, 10);
      X := Where_X;
      Y := Where_Y;
      Put ("Current point location is" & Integer'Image (X));
      Put_Line ("," & Integer'Image (Y));
      Draw_To (30, 30);
      Draw_To (50, 10, Light_Cyan);
      Draw_To (10, 10, Yellow);
   end Show_Poly_Draw;

   --------------------------------
   -- Display text demonstration --
   --------------------------------
   procedure Show_Display_Text is
   begin
      Wait_For_Keypress ("display some text");
      Clear_Window;
      Display_Text (10, 10, "The quick brown fox jumped over the lazy dog");
      Draw_Box (10, 30, 10 + 34 * X_Char - 1, 30 + Y_Char - 1, Blue, Fill);
      Display_Text (10, 30, "This is an example of colored text", Yellow);
      for I in 0 .. 7 loop
         Display_Text (10 + I, 50 + I * Y_Char,
            "This is line" & Integer'Image(I + 1),
            Color_Type'Val (I + 8));
      end loop;
   end Show_Display_Text;

   ----------------------------------
   -- Keyboard input demonstration --
   ----------------------------------
   procedure Show_Keyboard_Input is
      C : Character;
   begin
      Wait_For_Keypress ("get keyboard input");
      Clear_Window;
      Display_Text (0, 0, "Press a key:");
      while not Key_Hit loop
         null;
      end loop;
      C := Get_Key;
      if C /= Vk_Nul then
         Display_Text (0, Y_Char, "You pressed the '" & C & "' key.");
      else
         case (Get_Key) is
            when Vk_Prior =>
               Display_Text (0, Y_Char, "You pressed 'Page_Up'");
            when Vk_Next =>
               Display_Text (0, Y_Char, "You pressed 'Page_Down'");
            when Vk_End =>
               Display_Text (0, Y_Char, "You pressed 'End'");
            when Vk_Home =>
               Display_Text (0, Y_Char, "You pressed 'Home'");
            when Vk_Left =>
               Display_Text (0, Y_Char, "You pressed 'Cursor_Left'");
            when Vk_Up =>
               Display_Text (0, Y_Char, "You pressed 'Cursor_Up'");
            when Vk_Right =>
               Display_Text (0, Y_Char, "You pressed 'Cursor_Right'");
            when Vk_Down =>
               Display_Text (0, Y_Char, "You pressed 'Cursor_Dowm'");
            when Vk_Insert =>
               Display_Text (0, Y_Char, "You pressed 'Insert'");
            when Vk_Delete =>
               Display_Text (0, Y_Char, "You pressed 'Delete'");
            when others =>
               Display_Text (0, Y_Char, "You pressed an unknown key");
         end case;
      end if;
   end Show_Keyboard_Input;

   -------------------------------
   -- Mouse input demonstration --
   -------------------------------
   procedure Show_Mouse_Input is

      Size        : constant Integer    := 5;
      Back_Color  : constant Color_Type := Black;
      Norm_Color  : constant Color_Type := White;
      Left_Color  : constant Color_Type := Light_Blue;
      Right_Color : constant Color_Type := Light_Red;

      C       : Character;
      Mouse   : Mouse_Type;
      X, Y    : Integer := 0;
      Current : Color_Type := Norm_Color;

      procedure Adjust_Size is
      begin
         if (X - Size) < 0 then
            X := Size;
         end if;
         if (Y - Size) < 0 then
            Y := Size;
         end if;
         if (X + Size) > X_Max then
            X := X_Max - Size;
         end if;
         if (Y + Size) > Y_Max then
            Y := Y_Max - Size;
         end if;
      end Adjust_Size;

      procedure Write_Ball is
      begin
         Adjust_Size;
         Draw_Circle (X, Y, Size, Current, Fill);
      end Write_Ball;

      procedure Delete_Ball is
      begin
         Adjust_Size;
         Draw_Circle (X, Y, Size, Back_Color, Fill);
      end Delete_Ball;

   begin
      Wait_For_Keypress ("get mouse input");
      Clear_Window;
      Put_Line ("Press a button to change color");
      Put_Line ("Press any key in graphics window to quit");
      Write_Ball;
      while not Key_Hit loop
         if Mouse_Event then
            Mouse := Get_Mouse;
            Delete_Ball;
            case Mouse.Event is
               when None       => null;
               when Moved      => X := Mouse.X_Pos; Y := Mouse.Y_Pos;
               when Left_Up    => Current := Norm_Color;
               when Right_Up   => Current := Norm_Color;
               when Left_Down  => Current := Left_Color;
               when Right_Down => Current := Right_Color;
            end case;
            Write_Ball;
         end if;
      end loop;
      Clear_Window;
      C := Get_Key;
      if C = Vk_Nul then
         C := Get_Key;
      end if;
   end Show_Mouse_Input;

-----------------------
-- Main demo program --
-----------------------
begin

   Init_Display;

   Show_Max_Size;
   Show_Clear_Window;
   Show_Put_Pixel;
   Show_Draw_Line;
   Show_Draw_Box;
   Show_Get_Pixel;
   Show_Draw_Circle;
   Show_Draw_Ellipse;
   Show_Flood_Fill;
   Show_Poly_Draw;
   Show_Display_Text;
   Show_Keyboard_Input;
   Show_Mouse_Input;

   Kill_Display;

end Demo;
