-------------------------------------------------------------------------------
-- Program Adapaint                                                          --
-- (C) Copyright 1999 ADALOG                                                 --
-- Author: J-P. Rosen                                                        --
--                                                                           --
-- Rights to use, distribute or modify this package in any way is hereby     --
-- granted, provided this header is kept unchanged in all versions.          --
-- Additionnal headers may be added. If you make a valuable addition,        --
-- please keep us informed by sending a message to rosen.adalog@wanadoo.fr   --
--                                                                           --
-- ADALOG is providing training, consultancy and expertise in Ada and        --
-- related software engineering techniques. For more info about our services:--
-- ADALOG                   Tel: +33 1 46 45 51 12                           --
-- 27, avenue de Verdun     Fax: +33 1 46 45 52 49                           --
-- 92170 VANVES             E-m: rosen.adalog@wanadoo.fr                     --
--                          URL: http://pro.wanadoo.fr/adalog                --
--                                                                           --
-- This program is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY  --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                      --
-------------------------------------------------------------------------------
with AdaGraph; use AdaGraph;
with Text_IO;
procedure AdaPaint is
   Palet_Height : constant := 20;

   Sequence_End : exception;
   Command_End  : exception;

   -- Global variables
   X_Win,  Y_Win  : Natural;  -- Window size
   Y_Graph        : Natural;  -- Height of the graphic area (Palet removed)
   X_Char, Y_Char : Natural;  -- Character size

   Command           : Character;  -- Current command
   Prev_Command      : Character;  -- Previous command
   X_Start,  Y_Start : Natural;
   X_End,  Y_End     : Natural;
   Color             : Color_Type := Black;    -- Current color
   Filling           : Fill_Type  := No_Fill;  -- Current fill status

   Prev_Message   : String (1..20);   --  Message in header
   Message_Length : Natural := 0;     --  its length

   --
   -- Header
   -- Change window header
   -- If Message = "", keep previous message
   --
   procedure Header (Message : String := "") is
      Title : constant String := "Painting Program - "    &
                                 Color_Type'Image(Color)  & ", " &
                                 Fill_Type'Image(Filling) & " - ";
   begin
      if Message = "" then
         Set_Window_Title (Title & Prev_Message (1..Message_Length));
      else
         Set_Window_Title (Title & Message);
         Message_Length := Message'Length;
         Prev_Message (1..Message_Length) := Message;
      end if;
   end Header;

   --
   --  Draw_Palet
   --  (re)draws the palet at the bottom of the screen
   --
   procedure Draw_Palet is
      Palet_Width : constant Natural := X_Win / 8;
   begin
      for Pos in 0..7 loop
         Draw_Box (Pos*Palet_Width,            Y_Graph+1,
                   (Pos+1)*Palet_Width,        Y_Graph+Palet_Height,
                   Hue => Color_Type'Val(Pos), Filled => Fill);
      end loop;
      for Pos in 0..7 loop
         Draw_Box (Pos*Palet_Width,              Y_Graph+Palet_Height+1,
                   (Pos+1)*Palet_Width,          Y_Win-1,
                   Hue => Color_Type'Val(Pos+8), Filled => Fill);
      end loop;
   end Draw_Palet;

   --
   --  Help
   --  Prints instructions in the Dos window
   --
   procedure Help is
      use Text_IO;
   begin
      Put_Line ("Colors :");
      Put_Line ("   Left click  : select non-filled");
      Put_Line ("   Right click : select filled");
      New_Line;
      Put_Line ("Commands :");
      Put_Line ("   B: Brush");
      Put_Line ("   E: Ellipse");
      Put_Line ("   L: Line");
      Put_Line ("   H: Hand drawing");
      Put_Line ("   P: Pour");
      Put_Line ("   Q: Quit");
      Put_Line ("   R: Rectangle");
      Put_Line ("   S: Segments");
      Put_Line ("   T: Text (exit with escape)");
   end Help;

   --
   --  Wait
   --  Waits for Event to happen, or End_Event which raises Sequence_End
   --  If a Key is hit, it is a request for tool change, therefore it raises Command_End
   --  This procedure intercepts clicks in the palet and deals with color changes.
   --  Note: This is the ONLY place where we wait for events.
   --
   procedure Wait
     (Event     :     Event_Type;
      X, Y      : out Natural;
      End_Event :     Event_Type := None)
   is
      Current_Event : Mouse_Type;
   begin
      loop
         if Key_Hit then
            Prev_Command := Command;
            Command      := Get_Key;
            raise Command_End;
         end if;
         if Mouse_Event then
            Current_Event := Get_Mouse;
            if Current_Event.Y_Pos > Y_Graph then
               -- Event in palet
               case Current_Event.Event is
                  when Left_Down =>
                     Color   := Get_Pixel (Current_Event.X_Pos, Current_Event.Y_Pos);
                     Filling := No_Fill;
                     Header;
                  when Right_Down =>
                     Color   := Get_Pixel (Current_Event.X_Pos, Current_Event.Y_Pos);
                     Filling := Fill;
                     Header;
                  when others =>
                     null;
               end case;
            elsif Current_Event.Event = Event then
               X := Current_Event.X_Pos;
               Y := Integer'Min(Current_Event.Y_Pos, Y_Graph);
               return;
            elsif Current_Event.Event = End_Event then
               raise Sequence_End;
            end if;
         end if;
      end loop;
   end Wait;

   --
   --  Invert_Rectangle
   --  Draw a rectangle of dashed, inverted color for the selection
   --
   procedure Invert_Rectangle (X_Start, Y_Start, X_End, Y_End : in Integer) is
      Step  : constant := 4;
      X_Min : constant Integer := Integer'Min (X_Start, X_End);
      Y_Min : constant Integer := Integer'Min (Y_Start, Y_End);
      X_Max : constant Integer := Integer'Max (X_Start, X_End);
      Y_Max : constant Integer := Integer'Max (Y_Start, Y_End);
      X, Y  : Natural;

      function Invert (Color : Color_Type) return Color_Type is
         pragma InLine (Invert);
         Pivot : constant := Color_Type'Pos (Color_Type'Last);
      begin
         return Color_Type'Val(Pivot - Color_Type'Pos (Color));
      end Invert;

   begin
      X := X_Min;
      loop
         Put_Pixel (X, Y_Min, Invert (Get_Pixel(X, Y_Min)));
         exit when X >= X_Max - Step;
         X := X + Step;
      end loop;
      Y := Y_Min;
      loop
         Put_Pixel (X_Max, Y, Invert (Get_Pixel(X_Max, Y)));
         exit when Y >= Y_Max - Step;
         Y := Y + Step;
      end loop;

      if X_Min = X_Max or Y_Min = Y_Max then
         -- Not a true rectangle
         return;
      end if;

      X := X_Max;
      loop
         Put_Pixel (X, Y_Max, Invert (Get_Pixel(X, Y_Max)));
         exit when X <= X_Min + Step;
         X := X - Step;
      end loop;
      Y := Y_Max;
      loop
         Put_Pixel (X_Min, Y, Invert (Get_Pixel(X_Min, Y)));
         exit when Y <= Y_Min + Step;
         Y := Y - Step;
      end loop;
   end Invert_Rectangle;

   --
   --  Get_Selection
   --  Get the selection on the screen
   --  If Initialized = True, get the starting point from X_Min, Y_Min (for segments)
   --  If Initialized = False, the starting point is the place of first click.
   --
   procedure Get_Selection
      (X_Min, Y_Min : in out Natural;
       X_Max, Y_Max :    out Natural;
       Initialized  : in     Boolean := False)
   is
      X, Y    : Natural;
   begin
      Wait (Left_Down, X, Y);
      if not Initialized then
         X_Min := X;
         Y_Min := Y;
      end if;
      X_Max := X;
      Y_Max := Y;
      Invert_Rectangle (X_Min, Y_Min, X_Max, Y_Max);
      loop
         begin
            Wait (Moved, X, Y, End_Event => Left_Up);
         exception
            when Sequence_End =>
               Invert_Rectangle (X_Min, Y_Min, X_Max, Y_Max);
               return;
            when Command_End =>
               Invert_Rectangle (X_Min, Y_Min, X_Max, Y_Max);
               raise;
         end;
         Invert_Rectangle (X_Min, Y_Min, X_Max, Y_Max);
         X_Max := X;
         Y_Max := Y;
         Invert_Rectangle (X_Min, Y_Min, X_Max, Y_Max);
      end loop;
   end Get_Selection;

   -------------------------
   --  Drawing procedures --
   -------------------------

   --
   --  Enter_Segments
   --
   procedure Enter_Segments is
      X_Start, Y_Start, X_End, Y_End : Natural;
   begin
      Get_Selection (X_Start, Y_Start, X_End, Y_End);
      Goto_XY (X_Start, Y_Start);
      loop
         Draw_To (X_End, Y_End, Hue => Color);
         X_Start := X_End;
         Y_Start := Y_End;
         Get_Selection (X_Start, Y_Start, X_End, Y_End, Initialized => True);
      end loop;
   end Enter_Segments;

   --
   --  Enter_Hand_Held
   --
   procedure Enter_Hand_Held is
      X, Y : Natural;
   begin
      Wait (Left_Down, X, Y);
      Goto_XY (X, Y);
      Draw_To (X, Y, Hue => Color);
      loop
         Wait (Moved, X, Y, End_Event => Left_Up);
         Draw_To (X, Y, Hue => Color);
      end loop;
   exception
      when Sequence_End =>
         null;
   end Enter_Hand_Held;

   --
   --  Enter_Brush
   --
   procedure Enter_Brush is
      X, Y : Natural;
   begin
      Wait (Left_Down, X, Y);
      Draw_Box (Integer'Max (0, X-5),       Integer'Max (0, Y-5),
                Integer'Min (X_Win-1, X+5), Integer'Min (Y_Graph-1, Y+5),
                Hue => Color,               Filled => Fill);
      loop
         Wait (Moved, X, Y, End_Event => Left_Up);
         Draw_Box
           (Integer'Max (0, X-5),        Integer'Max (0, Y-5),
            Integer'Min (X_Win-1, X+5),  Integer'Min (Y_Graph, Y+5),
            Hue => Color,                Filled => Fill);
      end loop;
   exception
      when Sequence_End =>
         null;
   end Enter_Brush;

   --
   --  Enter_Text
   --
   procedure Enter_Text is
      X_Start, Y_Start : Natural;
      C   : Character;
      S   : String (1..80);
      Inx : Natural range 0..S'Last := 0;
   begin
      Wait (Left_Up, X_Start, Y_Start);
      if Y_Start + Y_Char >= Y_Win then
         return;  -- Too close to bottom
      end if;

      while Inx < S'Last loop
         C := Get_Key;

         if C = Ascii.Esc then
            raise Command_End;

         elsif C = Ascii.BS then
            Display_Text (X_Start, Y_Start, S(1..Inx) & ' ', Hue => White);
            if Inx > 0 then
               Inx := Inx - 1;
            end if;

         elsif C = Ascii.CR then
            Inx   := 0;
            Y_Start := Y_Start + Y_Char;
            if Y_Start + Y_Char >= Y_Win then
               return;  -- Too close to bottom
            end if;

         elsif X_Start + (Inx+1)*X_Char >= X_Win then
            -- No more space
            exit;

         else
            Inx := Inx + 1;
            S (Inx) := C;
         end if;
         Display_Text (X_Start, Y_Start, S(1..Inx), Hue => Color);
      end loop;
   end Enter_Text;

begin      -- Main program
   Help;

   Create_Graph_Window (X_Win, Y_Win, X_Char, Y_Char);
   Y_Graph := Y_Win - 2*Palet_Height - 1;
   Clear_Window (White);
   Draw_Palet;

   Command      := 'H';
   Prev_Command := Command;
   loop
      begin
         case Command is
            when 'b' | 'B' => -- Brush
               Header ("Brush");
               Enter_Brush;
            when 'e' | 'E' => -- Ellipse
               Header ("Ellipse");
               Get_Selection (X_Start, Y_Start, X_End, Y_End);
               Draw_Ellipse (X_Start, Y_Start, X_End, Y_End,
                  Hue => Color, Filled => Filling);
            when 'l' | 'L' => -- Line
               Header ("Line");
               Get_Selection (X_Start, Y_Start, X_End, Y_End);
               Draw_Line (X_Start, Y_Start, X_End, Y_End, Hue => Color);
            when 'h' | 'H' => -- Hand held drawing
               Header ("Hand_Held drawing");
               Enter_Hand_Held;
            when 'p' | 'P' => -- Pour
               Header ("Pour");
               Wait (Left_Down, X_Start, Y_Start);
               Wait (Left_Up, X_Start, Y_Start);
               Flood_Fill (X_Start, Y_Start, Color);
               Draw_Palet;  -- If ever "Pour" destroyed the palet
            when 'q' | 'Q' => -- Quit
               exit;
            when 'r' | 'R' => -- Rectangle
               Header ("Rectangle");
               Get_Selection (X_Start, Y_Start, X_End, Y_End);
               Draw_Box (X_Start, Y_Start, X_End, Y_End,
                         Hue => Color, Filled => Filling);
            when 's' | 'S' => -- Segments
               Header ("Segments");
               Enter_Segments;
            when 't' | 'T' => -- Text
               Header ("Text");
               Enter_Text;
            when others => -- Unknown
               Command := Prev_Command;
         end case;
      exception
         when Command_End =>
            null;
      end;
   end loop;

   Destroy_Graph_Window;
end AdaPaint;