-----------------------------------------------------------------------
--
--  File:        adagraph.adb
--  Description: basic X11 graphics
--  Rev:         0.2
--  Date:        02-may-2000
--  Prel:        X11Ada - bindings to X11 from www.gnuada.org
--  Author:      Dr. Gerald Kasner
--  Mail:        Gerald.Kasner@Physik.Uni-Magdeburg.de
--
--  Copyright (c) Jerry van Dijk, 1997, 1998, 1999
--  Billie Hollidaystraat 28
--  2324 LK Leiden
--  THE NETHERLANDS
--  tel int +31 (0)71 531 4365
--  for the win32 spec
--
--  Copyright (c) Dr. Gerald Kasner 1999, 2000  
--  Kiefholzstrasse 15
--  39340 Haldensleben
--  Germany
--  for the X11 body
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------


   with Text_IO, X, X.Xlib, X.Strings, X.Xutil, Interfaces.C, 
   Interfaces.C.Strings, Interfaces.C.Pointers, Unchecked_Deallocation,
   Ada.Strings.Unbounded, System, Unchecked_Conversion; 
   use  Text_IO, X, X.Xlib, X.Strings, X.Xutil, Interfaces.C,
   Interfaces.C.Strings, Ada.Strings.Unbounded;


   package body adagraph is
   
      Package I_IO is new Integer_IO(integer);
   
   ----------------------
   -- Global variables --
   ----------------------
   
      Adagraph_X11_Version: constant integer:= 2;
      MaxColors           : constant integer:=16;
   
      rgbcolor_data   : constant array(color_type,1..3) of X.unsigned_short:=
      ((     0,     0,     0),  -- Black
      (     0,     0, 32511),  -- Blue
      (     0, 32511,     0),  -- Green
      (     0, 32511, 32511),  -- Cyan
      ( 32511,     0,     0),  -- Red
      ( 32511,     0, 32511),  -- Magneta
      ( 32511, 32511,     0),  -- Brown
      ( 44287, 44287, 44287),  -- Light_Gray
      ( 20479, 20479, 20479),  -- Dark_Gray
      (     0,     0, 65535),  -- Light_Blue
      (     0, 65535,     0),  -- Light_Green
      (     0, 65535, 65535),  -- Light_Cyan
      ( 65535,     0,     0),  -- Light_Red
      ( 65535,     0, 65535),  -- Light_Magneta
      ( 65535, 65535,     0),  -- Yello
      ( 65535, 65535, 65535)); -- White                    
   
      display         : X.Xlib.XDisplay_access := null;
      key_avail       : boolean :=false;
      drawable1       : X.Drawable;
      win,root        : X.Window;
      gc              : X.Xlib.GC;
      colormap        : X.Colormap;
      width,maxwidth,
      height,maxheight,
      screen          : X.signed_int;
      visual          : X.Xlib.Visual_access:=null;
   
      Default_Window_Name : constant String := "AdaGraph-X";
      Font                : constant String := "9x15";
      colors              : array (color_type) of X.unsigned_long;
   
      type polydraw_point is record
            x,y : integer;
         end record;
   
      pd_point : polydraw_point;
      key      : character:=character'first;   
      mouse    : Mouse_Type;
      old_mask_return : X.Unsigned_Int:=0;  
   
      image    : X.Xlib.XImage_access;
   
      type windowtype is (default,sized,ext);
   
   ----------------------------------
   -- System information functions --
   ----------------------------------
   
   ---------------------
   -- Get_Dll_Version --
   ---------------------
   
      function Get_Dll_Version return Integer is
      begin
         return Adagraph_X11_Version;
      end Get_Dll_Version;
   
   ------------------
   -- Get_Max_Size --
   ------------------
   
      procedure Get_Max_Size (X_Size, Y_Size : out Integer) is
      begin
         maxwidth:=DisplayWidth(display,screen);
         maxheight:=DisplayHeight(display,screen);
         X_Size:=Integer(maxwidth);
         Y_Size:=Integer(maxheight);
      end Get_Max_Size;
   
   -----------------------
   -- Window management --
   -----------------------   
   
   ------------------------------
   -- Main Window Procedure    --
   -- not visible from outside --
   ------------------------------
   
      procedure Graph_Window
      (X_Size, Y_Size : in     Integer;
      X_Max,  Y_Max  :    out Integer;
      X_Char, Y_Char :    out Integer;
      wintype        : in     windowtype) is
      type Bitmap is array(natural range <>) of Interfaces.C.Unsigned_Char;
          icon_bitmap_bits: constant bitmap := (
       16#00#, 16#00#, 16#00#, 16#00#, 16#10#, 16#b8#, 
       16#10#, 16#cc#, 16#28#, 16#84#, 16#28#, 16#02#, 
       16#44#, 16#c2#, 16#fc#, 16#9e#, 16#44#, 16#8c#, 
       16#82#, 16#cc#, 16#c7#, 16#7b#, 16#00#, 16#03#, 
       16#80#, 16#06#, 16#80#, 16#04#, 16#40#, 16#08#, 
       16#e0#, 16#1c#);
  
       function to_cptr is new Unchecked_Conversion(
           System.Address, X.Strings.const_charp);

         wattr      : aliased X.Xlib.XsetWindowAttributes;
         event      : aliased X.Xlib.XEvent;
         iconPixmap : X.Pixmap;                   
         sizehints  : aliased X.Xutil.XSizeHints;
         class_hint : aliased X.Xutil.XClassHint;
         xmx,ymx    : Integer;
         ww,hh      : Unsigned_Int;
         values     : aliased X.Xlib.XGCValues;
         font_info  : X.Xlib.XFontStruct_access;
         resname    : constant String:="AdaGraph";
         classname  : constant String:="AdaGraph-X";
         st         : X.Xlib.Status;
         col        : aliased X.Xlib.XColor;
      begin 
         display:=XOpenDisplay(Null);
         screen :=DefaultScreen(Display);
         root:=RootWindow(display,screen);
      
         wattr.event_mask:=ExposureMask;
      
         Get_Max_Size(xmx,ymx);
      
         case wintype is
            when default=> ww:=Unsigned_Int(xmx/2);
               hh:=Unsigned_Int(ymx/2);
            when sized  => ww:=Unsigned_Int(X_Size);
               hh:=Unsigned_Int(Y_Size);
            when ext    => ww:=Unsigned_Int(xmx);
               hh:=Unsigned_Int(ymx);
         end case;
      
         X_Max:=Integer(ww);
         Y_Max:=Integer(hh);
         width:=Interfaces.C.Int(ww);
         height:=Interfaces.C.Int(hh);
      
         visual:=XDefaultVisual(display,screen);
         win:=XCreateWindow(display,root,0,0,ww,hh,0,
            CopyFromParent,InputOutPut,visual,
            CWEventMask,wattr'access);
      
         colormap:=DefaultColorMap(display,screen);
         drawable1:=X.Drawable(win);
         gc:=XCreateGC(display,drawable1,0,values'access);
      
         for i in color_type'range loop
            col.red   := rgbcolor_data(i,1);
            col.green := rgbcolor_data(i,2);
            col.blue  := rgbcolor_data(i,3);
            st:=XAllocColor(display,colormap,col'access);
            colors(i):=col.pixel;
         end loop;
      
         iconPixmap:=XCreateBitmapFromData(display,drawable(win),
                     to_cptr(icon_bitmap_bits(0)'Address),16,16);
         
      
         sizehints.min_width:=signed_int(ww);         
         sizehints.max_width:=signed_int(ww);
         sizehints.min_height:=signed_int(hh);
         sizehints.max_height:=signed_int(hh);
      
         pd_point.x:=integer(ww/2);
         pd_point.y:=integer(hh/2);  
      
         sizehints.flags:=PMinSize + PMaxSize; 
      
         XSetStandardProperties(display,win,
            Addr(Default_Window_Name),Addr(Default_Window_Name),
            iconpixmap,Null,0,sizehints'access);
      
         class_hint.res_name:=addr(resname);
         class_hint.res_class:=addr(classname);
      
         XSetClassHint(display,win,class_hint'access); 
      
         XmapRaised(display,win);
      
         XSetWindowBackground(display,win,WhitePixel(display,screen));
         XClearWindow(display,win);
      
         font_info:=XLoadQueryFont(display,addr(Font));
         X_Char:=9; -- for now,
         Y_Char:=13;--integer(font_info.ascent+font_info.descent);-- to be refined
      
         XNextEvent(display,Event'access);
         while Event.Event_Type /= Expose loop
            XNextEvent(display,Event'access);
         end loop;
      
         wattr.event_mask:=Interfaces.C.Long(Interfaces.C.Unsigned(X.ButtonPressMask) or 
                   Interfaces.C.Unsigned(X.ButtonReleaseMask) or
                   Interfaces.C.Unsigned(X.KeyPressMask) or
                   Interfaces.C.Unsigned(X.PointerMotionMask));
    
         wattr.backing_store:=Always;
      
         XChangeWindowAttributes(display,win,CWEventMask or CWBackingStore,wattr'access);
      
         XFlush(display);
      
      end Graph_Window;
   
   
   -------------
   -- Is_Open --
   -------------
   
      function Is_Open return Boolean is
      begin
         if display=null then 
            return false;
         else 
            return true;
         end if;
      end Is_Open;
   
   ----------------------
   -- Set_Window_Title --
   ----------------------
   
      procedure Set_Window_Title (Title : in String) is
      begin        
         XStoreName(display,win,addr(Title));
         XFlush(Display);
      end Set_Window_Title;
   
   ------------------
   -- Clear_Window --
   ------------------
   
      procedure Clear_Window (Hue : in Color_Type := Black) is
      begin       
         XSetForeground(display,gc,colors(hue));
         XFillRectangle(display,drawable1,gc,0,0,
            Interfaces.C.Unsigned(width),
            Interfaces.C.Unsigned(height));
         XFlush(display);
      end Clear_Window;
   
   
   -------------------------------
   -- Create_Sized_Graph_Window --
   -------------------------------
   
      procedure Create_Sized_Graph_Window
      (X_Size, Y_Size : in     Integer;
      X_Max,  Y_Max  :    out Integer;
      X_Char, Y_Char :    out Integer)
      is
      begin
         Graph_Window(X_Size,Y_Size,X_Max,Y_Max,X_Char,Y_Char,sized);
         Set_Window_Title(Default_Window_Name);
         XFlush(display);
      end Create_Sized_Graph_Window;
   
   -------------------------
   -- Create_Graph_Window --
   -------------------------
   
      procedure Create_Graph_Window
      (X_Max,  Y_Max  : out Integer;
      X_Char, Y_Char : out Integer)
      is
      begin
         Graph_Window(0,0,X_Max,Y_Max,X_Char,Y_Char,default);
         Set_Window_Title(Default_Window_Name);
         XFlush(display);
      end Create_Graph_Window;
   
   -----------------------------
   -- Ext_Create_Graph_Window --
   -----------------------------
   
      procedure Ext_Create_Graph_Window
      (X_Max,  Y_Max  : out Integer;
      X_Char, Y_Char : out Integer)
      is
      begin
         Graph_Window(0,0,X_Max,Y_Max,X_Char,Y_Char,ext);
         Set_Window_Title(Default_Window_Name);
         XFlush(display);
      end Ext_Create_Graph_Window;
   
   --------------------------
   -- Destroy_Graph_Window --
   --------------------------
   
      procedure Destroy_Graph_Window is
      begin
         XUnmapWindow(display,win);
         XDestroyWindow(display,win);
         XDestroyWindow(display,root);
         XFreeGC(display,gc);
         XFreeColormap(display,colormap);
         XCloseDisplay(display);
         width:=0;
         height:=0;
         display:=null;
      end Destroy_Graph_Window;
   
   ---------------------
   -- Input functions --
   ---------------------
   
   -------------
   -- Key_Hit --
   -------------
   
      function Key_Hit return Boolean is
      
      ret   : X.Xlib.Bool;
      Event : aliased X.Xlib.Xevent;
      ks    : X.KeySym;
      ptr   : X.Strings.charp;
      ss    : Unbounded_String;
         begin
         key:=character'first;
         ret:=XCheckMaskEvent(display,X.KeyPressMask,Event'access);
         if ret>0 then
            case event.event_type is
                     when KeyPress => 
                          ks:=XKeycodeToKeySym(display,event.xkey.keycode,0);
                          ss:=To_Unbounded_String(
                              Interfaces.C.Strings.Value(
                              X.Strings.to_chars_ptr(XKeysymToString(ks))));
                          if Length(ss)=1 then 
                             key:=Element(ss,1);
--                           else
--                              if To_String(ss)="Prior" then
--                                 key:=Vk_Prior;
--                              end if;
--                              if To_String(ss)="Next" then
--                                 key:=Vk_Next;
--                              end if;
--    should be                 if To_String(ss)="End" then
--    refined,                  key:=Vk_End;
--    contact Jerry             end if;
--    G.K. 04 May 2000          if To_String(ss)="Home" then
--                                 key:=Vk_Home;
--                              end if;
--                              if To_String(ss)="Left" then
--                                 key:=Vk_Left;
--                              end if;
--                              if To_String(ss)="Right" then
--                                 key:=Vk_Right;
--                              end if;
--                              if To_String(ss)="Down" then
--                                 key:=Vk_Down;
--                              end if;
--                              if To_String(ss)="Insert" then
--                                 key:=Vk_Insert;
--                              end if;
--                              if To_String(ss)="Delete" then
--                                 key:=Vk_Delete;
--                              end if;
                          end if;    
                     when others=>null;                                              
            end case;
            return true;
         else
            return false;
         end if;           
      end Key_Hit;
   
   -------------
   -- Get_Key --
   -------------
   
      function Get_Key return Character is
      begin   
          return key;             
      end Get_Key;
   
   -----------------
   -- Mouse_Event --
   -----------------
   
      function Mouse_Event return Boolean is
      ret   : X.Xlib.Bool;
      event : aliased X.Xlib.XEvent;
      begin
         ret:=XCheckMaskEvent(display,Interfaces.C.long(
                             Interfaces.C.Unsigned(X.ButtonPressMask) or 
                             Interfaces.C.Unsigned(X.ButtonReleaseMask) or
                             Interfaces.C.Unsigned(X.PointerMotionMask)),event'access);
         if ret>0 then
            case event.event_type is
            when ButtonPress => 
               case event.xbutton.button is
                  when button1=>mouse.X_Pos:=Integer(event.xbutton.xx);
                     mouse.Y_Pos:=Integer(event.xbutton.y);
                     mouse.Event:=Left_Down;
                  when button3=>mouse.X_Pos:=Integer(event.xbutton.xx);
                     mouse.Y_Pos:=Integer(event.xbutton.y);
                     mouse.Event:=Right_Down;
                  when others=>null;
               end case;               
            when ButtonRelease => 
               case event.xbutton.button is
                  when button1=>mouse.X_Pos:=Integer(event.xbutton.xx);
                     mouse.Y_Pos:=Integer(event.xbutton.y);
                     mouse.Event:=Left_Up;
                  when button3=>mouse.X_Pos:=Integer(event.xbutton.xx);
                     mouse.Y_Pos:=Integer(event.xbutton.y);
                     mouse.Event:=Right_Up;
                  when others=>null;
               end case;
            when MotionNotify => mouse.X_Pos:=Integer(event.xmotion.xx);
               mouse.Y_Pos:=Integer(event.xmotion.y);
               mouse.Event:=Moved;                                          
            when others => null;                                               
            end case;
            return true;
         else
            mouse.X_Pos:=0;
            mouse.Y_Pos:=0;
            mouse.Event:=None;
            return false;
         end if;

         
      end Mouse_Event;
   
   ---------------
   -- Get_Mouse --
   ---------------
   
      function Get_Mouse return Mouse_Type is
      begin
       return mouse;
      end Get_Mouse;
   
   -----------------------
   -- Graphic functions --
   -----------------------
   
   -----------------------------------------------------
   -- Clear_Window moved to window management section --
   -- G.K. Dec. 01 1999                               --
   -----------------------------------------------------
   
   ---------------
   -- Get_Pixel --
   ---------------
   
      function Get_Pixel (X, Y : in Integer) return Color_Type is
         plane_mask : Interfaces.C.Unsigned_Long:=4294967295;
         pixel      : Interfaces.C.Unsigned_Long:=0;
         colort: Color_Type:=Black;
         st    : signed_int;
      begin
         Image:=XGetImage(display,drawable1,
            Interfaces.C.Int(X),
            Interfaces.C.Int(Y),
            1,1,plane_mask,ZPixmap);
      
         pixel:=XGetPixel(Image,0,0);
      
         st:=XDestroyImage(Image);
         for col in Color_Type'range loop
            if pixel=colors(col) then
               colort:=col;
            end if;
         end loop;   
      
         XFlush(display);    
         return colort;
      
      end Get_Pixel;
   
   ---------------
   -- Put_Pixel --
   ---------------
   
      procedure Put_Pixel (X, Y : in Integer; Hue : in Color_Type := White) is
      begin
         XSetForeground(display,gc,colors(hue));
         XDrawPoint(display,drawable1,gc,signed_int(X),signed_int(y));
         XFlush(display);
      end Put_Pixel;
   
   ---------------
   -- Draw_Line --
   ---------------
   
      procedure Draw_Line
      (X1, Y1, X2, Y2 : in Integer;
      Hue            : in Color_Type := White)
      is
      begin
         XSetForeground(display,gc,colors(hue));
         XDrawLine(display,drawable1,gc,signed_int(X1),signed_int(Y1),
            signed_int(X2),signed_int(Y2));
         XFlush(display); 
      end Draw_Line;
   
   --------------
   -- Draw_Box --
   --------------
   
      procedure Draw_Box
      (X1, Y1, X2, Y2 : in Integer;
      Hue            : in Color_Type := White;
      Filled         : in Fill_Type  := No_Fill) is
         width,height: Interfaces.C.unsigned;
      begin
         XSetForeground(display,gc,colors(hue));
         width:=Interfaces.C.unsigned(x2-x1);
         height:=Interfaces.C.unsigned(y2-y1);
         XDrawRectangle(display,drawable1,gc,
            signed_int(X1),signed_int(Y1),
            width,height);
         if Filled /= No_Fill then
            width:=width+1;
            height:=height+1;
            XFillRectangle(display,drawable1,gc,
               signed_int(X1),signed_int(Y1),
               width,height);
         end if;     
         XFlush(display);
      end Draw_Box;
   
   -----------------
   -- Draw_Circle --
   -----------------
   
      procedure Draw_Circle
      (X, Y, Radius : in Integer;
      Hue          : in Color_Type := White;
      Filled       : in Fill_Type  := No_Fill)
      is
         r2:unsigned_int;
      begin
         r2:=unsigned_int(radius+radius);
         XSetForeground(display,gc,colors(hue));
         if Filled=fill then
            XFillArc(display,drawable1,gc,signed_int(X-Radius),signed_int(Y-Radius),
               r2,r2,0,23040);
         else
            XDrawArc(display,drawable1,gc,signed_int(X-Radius),signed_int(Y-Radius),
               r2,r2,0,23040);
         end if;
         XFlush(display);
      end Draw_Circle;
   
   ------------------
   -- Draw_Ellipse --
   ------------------
   
      procedure Draw_Ellipse
      (X1, Y1, X2, Y2 : in Integer;
      Hue            : in Color_Type := White;
      Filled         : in Fill_Type  := No_Fill)
      is
      begin
         XSetForeground(display,gc,colors(hue));
         if filled=fill then
            XFillArc(display,drawable1,gc,signed_int(X1),signed_int(y1),
               unsigned_int(X2-X1),unsigned_int(Y2-Y1),0,23040);
         else
            XDrawArc(display,drawable1,gc,signed_int(X1),signed_int(y1),
               unsigned_int(X2-X1),unsigned_int(Y2-Y1),0,23040);   
         end if;     
         XFlush(display);    
      end Draw_Ellipse;
   
   ----------------
   -- Flood_Fill --
   ----------------
   
      procedure Flood_Fill (X, Y : in Integer; Hue : in Color_Type := White) is
         xmx,ymx        : Integer;
         plane_mask     : Interfaces.C.Unsigned_Long:=4294967295;
         pixel,oldpixel : Interfaces.C.Unsigned_Long:=0;
         st    : signed_int;
            
         type Point is
            record
               x,y:Integer;
            end record;
      
         seed : Point;
      
         type direction is (left,right,up,down); 
      
         type Node;
         type StackOfPoints is access Node;
         type Node is 
            record
               pt  : Point;
               next: StackOfPoints;
            end record;
      
         procedure deallocate is new Unchecked_Deallocation(Node,StackOfPoints);
      
         function empty(s : in StackOfPoints) return boolean is
         begin
            return s=null;
         end empty;
      
         procedure push(inpt : in Point; s : in out StackOfPoints) is
            temp : StackOfPoints;
         begin
            temp:=new node'(inpt,s);
            s:=temp;
         end push;
      
         procedure pop(outpt : out Point; s : in out StackOfPoints) is
            temp : StackOfPoints;
         begin
            outpt:=s.pt;
            temp:=s;
            s:=s.next;
            deallocate(temp);
         end pop;
      
         function FindEnd(BeginPoint : in Point; Way : in direction) return Point is
            xinc      : integer;
            EndPoint  : Point;
         begin
            if Way=left then
               xinc:=-1; 
            else 
               xinc:= 1;
            end if;
         
            EndPoint:=BeginPoint;
            EndPoint.x:=EndPoint.x+xinc;
         
            while (XgetPixel(Image,signed_int(EndPoint.x),signed_int(EndPoint.y))=oldpixel
            and (EndPoint.x>=0) and (EndPoint.x<Integer(width))) loop
               EndPoint.x:=EndPoint.x+xinc;
            end loop;
         
            EndPoint.x:=EndPoint.x-xinc;
            return EndPoint;
         
         end FindEnd;
      
         function FindOtherEnd(BeginPoint : in Point; Way : in direction) return Point is
            xinc      : integer;
            EndPoint  : Point;
         begin
            if Way=left then
               xinc:=-1; 
            else 
               xinc:= 1;
            end if;
         
            EndPoint:=BeginPoint;
            EndPoint.x:=EndPoint.x+xinc;
         
            while (XgetPixel(Image,signed_int(EndPoint.x),signed_int(EndPoint.y))/=oldpixel
            and (EndPoint.x>=0) and (EndPoint.x<Integer(width))) loop
               EndPoint.x:=EndPoint.x+xinc;
            end loop;
         
            EndPoint.x:=EndPoint.x-xinc;
            return EndPoint;
         
         end FindOtherEnd;            
      
         procedure StackSeedPoints(Start, RightEnd : in Point; 
         UpOrDown  : in direction;
         PointStack: in out StackOfPoints) is
            CurrentPoint,OtherEnd : Point;     
            InColorRegion         : boolean;
         begin
            Currentpoint.x:=Start.x;
            if UpOrDown=down then
               CurrentPoint.y:=Start.y+1;
            else
               Currentpoint.y:=Start.y-1;
            end if;
            if XGetPixel(Image,signed_int(CurrentPoint.x),signed_int(CurrentPoint.y))=oldpixel then
               Currentpoint:=FindEnd(CurrentPoint,Left);
               Push(CurrentPoint,PointStack);
               CurrentPoint:=FindEnd(CurrentPoint,Right);
               CurrentPoint.x:=Currentpoint.x+1;
            end if;
            OtherEnd:=CurrentPoint;
            InColorRegion:=false;
            while OtherEnd.x < RightEnd.x loop
               if InColorRegion then
                  OtherEnd:=FindEnd(CurrentPoint,Right);
                  Push(CurrentPoint,PointStack);
                  InColorRegion:=False;
               else
                  OtherEnd:=FindOtherEnd(CurrentPoint,Right);
                  InColorRegion:=True;
               end if;
               if (OtherEnd.x<=RightEnd.x) or InColorRegion then
                  CurrentPoint.x:=OtherEnd.x+1;
               end if;
            end loop;
         end StackSeedPoints;
      
         procedure HLine(p1,p2: in Point) is
            st : signed_int;
         begin
            for i in p1.x..p2.x loop
               st:=XPutPixel(Image,signed_int(i),signed_int(p1.y),pixel);
            end loop;
         end Hline;
      
         procedure ScanLineSeedFill(seed : in Point) is
            CurrentPoint, RightEnd : Point;
            PointStack : StackOfPoints :=Null;
         begin
            if (XGetPixel(Image,signed_int(seed.x),signed_int(seed.y))=oldpixel)
            and (oldpixel/=pixel) then
               Currentpoint:=FindEnd(seed,Left);
               Push(CurrentPoint,PointStack);
               while not empty(Pointstack) loop
                  Pop(CurrentPoint,PointStack);
                  RightEnd:=FindEnd(CurrentPoint,Right);
                  if (CurrentPoint.y)>=0 then
                     StackSeedPoints(CurrentPoint,RightEnd,Up,PointStack);
                  end if;
                  if (CurrentPoint.y)<ymx then
                     StackSeedPoints(CurrentPoint,RightEnd,Down,PointStack);
                  end if;
                  HLine(CurrentPoint,RightEnd);
               end loop;
            end if;
         end ScanLineSeedFill;    
      
      
      begin
       
         Image:=XGetImage(display,drawable1,0,0,
            unsigned_int(width),
            unsigned_int(height), 
            plane_mask,ZPixmap);
      
         oldpixel:=XGetPixel(Image,signed_int(x),signed_int(y));
         pixel:=Colors(hue);
      
      
         xmx:=integer(width);  
         ymx:=integer(height); 
      
         seed.x:=X;
         seed.y:=Y;
      
         ScanLineSeedFill(seed);
      
         XPutImage(display,drawable1,gc,Image,0,0,0,0,
            unsigned_int(width),unsigned_int(height));
      
         st:=XDestroyImage(Image);
         XFlush(display);

      end Flood_Fill;
   
   ------------------
   -- Display_Text --
   ------------------
   
      procedure Display_Text
      (X, Y : in Integer;
      Text : in String;
      Hue  : in Color_Type := White)
      is
      begin
         XSetForeground(display,gc,colors(hue));
         XDrawString(display,drawable1,gc,
            signed_int(x),signed_int(y),
            Addr(Text),signed_int(Text'last));
         XFlush(display);
      end Display_Text;
   
   
   ----------------------
   -- Polydraw support --
   ----------------------
   
   -------------
   -- Where_X --
   -------------
   
      function Where_X return Integer is
      begin
         return pd_point.x;
      end Where_X;
   
   -------------
   -- Where_Y --
   -------------
   
      function Where_Y return Integer is
      begin
         return pd_point.y;      
      end Where_Y;
   
   -------------
   -- Goto_XY --
   -------------
   
      procedure Goto_XY (X, Y : in Integer) is
      begin
         pd_point.x:=X;
         pd_point.y:=Y;
      end Goto_XY;
   
   -------------
   -- Draw_To --
   -------------
   
      procedure Draw_To (X, Y : in Integer; Hue : in Color_Type := White) is
      begin
         Draw_Line(pd_point.x,pd_point.y,X,Y,Hue); 
         pd_point.x:=X;
         pd_point.y:=Y;       
      end Draw_To;
   
   end adagraph;

