with System ;
with Interfaces ;

with Storage ;
------------------------------------------------------------------------------
--
-- Abstract : Encrypt
--       This package is a base package for implementing various encryption
-- algorithms.
--
------------------------------------------------------------------------------
package Encrypt is

   -- One Way Hash Functions or Message Digest Algorithms
   -- Reference : 
   --     Applied Cryptography
   --     Protocols, Algorithms and Source Code in C
   --                - Bruce Schneier
   -- For each new algorithm, a new context type derived from OneWayHash_Context_T
   -- is defined and used as the basis for dispatching calls.

   type Signature_T is new Storage.Byte_Array_T ;
   type Onewayhash_Context_T is abstract tagged private ;

   -- This abstract procedure should initialize at least 
   -- the block size and the signature size records of the context
   procedure Initialize( Ctx : out Onewayhash_Context_T ) is abstract ;
   procedure Finalize( Ctx : in out Onewayhash_Context_T'Class ) ;
   function  HashLength( ctx : in OneWayHash_Context_T'Class ) return integer ;
   procedure Dump( Ctx : in OneWayHash_Context_T ) ;
   -- The Dispatcher will always send the data in the block size specified by
   -- the algorithm in the initialization. However only "data_size" bytes can be
   -- considered valid data. The last and final block is indicated by a block of
   -- 0 <= data_size < blocksize
   procedure Update( Ctx : in out Onewayhash_Context_T ;
         Data : Storage.Byte_Array_T ;
         Data_Size : in Integer ) is abstract ;
                     
   -- After the last block this function returns the signature of the digested
   -- amount
   function Onewayhash( Ctx : in Onewayhash_Context_T ) return Signature_T
                         is abstract ;
                     
   -- Digests the given file and returns the signature of the file
   procedure Generate( Filename : in String ;
         Context : in out Onewayhash_Context_T'Class ) ;
                    
   -- Digests the given block and returns the signature of the file
   procedure Generate( Block : in Storage.Byte_Array_T ;
                       Context : in out Onewayhash_Context_T'Class ) ;
   procedure Generate( Ptr : in System.Address ;
                       Blocksize : in Integer ;
                       Context : in out Onewayhash_Context_T'Class ) ;
                    
   procedure Print( Sig : in Signature_T ) ;
   function  Image( Sig : in Signature_T ) return String ;
   
private

   type Onewayhash_Context_T is abstract tagged
   record
      Initialized : Boolean := False ;
      finalized : boolean := false ;
      Block_Size : Integer := 0 ;
      Signature_Size : Integer := 0 ;
      Byte_Count : Interfaces.Unsigned_64 := 0 ;
   end record ;

end Encrypt ;

