with System.Storage_Elements ;

package body Encrypt.Ripemd160 is

   use Interfaces ;

   function F1( X , Y , Z : in Interfaces.Unsigned_32 ) 
	       return Interfaces.Unsigned_32 is
   begin
      return (X xor Y) xor Z ;
   end F1 ; 
   pragma Inline(F1) ;

   function F2( X , Y , Z : in Interfaces.Unsigned_32 )
	       return Interfaces.Unsigned_32 is
   begin
      return ( (X and Y) or ((not X) and Z)) ;
   end F2 ;
   pragma Inline(F2) ;

   function F3( X , Y , Z : in Interfaces.Unsigned_32 )
	       return Interfaces.Unsigned_32 is
   begin
      return ((X or (not Y)) xor Z) ;
   end F3 ;
   pragma Inline(F3) ;

   function F4( X , Y , Z : in Interfaces.Unsigned_32 )
	       return Interfaces.Unsigned_32 is
   begin
      return ( (X and Z) or (Y and (not Z))) ;
   end ;
   pragma Inline(F4) ;

   function F5( X , Y , Z : in Interfaces.Unsigned_32 )
	       return Interfaces.Unsigned_32 is
   begin
      return (X xor (Y or (not Z))) ;
   end F5 ;
   pragma Inline(F5) ;

   function Rol( X : in Interfaces.Unsigned_32 ; N : in Natural ) return Interfaces.Unsigned_32 is
   begin
      return Interfaces.Shift_Left(X,N) or Interfaces.Shift_Right(X,32-N) ;
   end Rol ;
   pragma Inline( Rol ) ;

   procedure FF1( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
                  S : in Natural ) is
   begin
      A := A + F1( B , C , D ) + X ;
      A := Rol( A , S ) + E ;
      C := Rol( C , 10 ) ;
   end Ff1 ;

   procedure FF2( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
                  S : in Natural ) is
   begin
      A := A + F2( B , C , D ) + X + 16#5a82_7999# ;
      A := Rol( A , S ) + E ;
      C := Rol( C , 10 ) ;
   end Ff2 ;

   procedure Ff3 ( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
		   S : in Natural ) is
   begin
      A := A + F3( B , C , D ) + X + 16#6ed9_eba1# ;
      A := Rol( A , S ) + E ;
      C := Rol( C , 10 ) ;
   end Ff3 ;

   procedure Ff4( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
		  S : in Natural ) is
   begin
      A := A + F4( B , C , D ) + X + 16#8f1b_bcdc# ;
      A := Rol( A , S ) + E ;
      C := Rol( C , 10 ) ;
   end Ff4 ;
   procedure Ff5( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
		  S : in Natural ) is
   begin
       A := A + F5( B , C , D ) + X + 16#A953_fd4e# ;
       A := Rol( A , S ) + E ;
       C := Rol( C , 10 ) ;
   end Ff5 ;

   procedure Fff1( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
		   S : in Natural ) is
   begin
       A := A + F1(B,C,D) + X ;
       A := Rol(A , S) + E ;
       C := Rol(C,10) ;
   end Fff1 ;

   procedure Fff2( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
		   S : in Natural ) is
   begin
      A := A + F2(B,C,D) + X + 16#7a6d_76e9# ;
      A := Rol(A,S) + E ;
      C := Rol(C,10) ;
   end Fff2 ;

   procedure Fff3( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
		   S : in Natural ) is
   begin
      A := A + F3(B,C,D) + X + 16#6d70_3ef3# ;
      A := Rol(A,S) + E ;
      C := Rol(C,10) ;
   end Fff3 ;

   procedure Fff4( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
		   S : in Natural ) is
   begin
      A := A + F4(B,C,D) + X + 16#5c4d_d124# ;
      A := Rol(A , S ) + E ;
      C := Rol( C, 10 ) ;
   end Fff4 ;

   procedure Fff5( A , B , C , D , E , X : in out Interfaces.Unsigned_32 ;
		   S : Natural ) is
   begin
      A := A + F5(B,C,D) + X + 16#50a2_8be6# ;
      A := Rol(A , S ) + E ;
      C := Rol(C,10) ;
   end Fff5 ;

   procedure Initialize( Ctx : out Context_T ) is
   begin
      Ctx.Block_Size := Block_Size ;
      Ctx.Byte_Count  := 0 ;
      Ctx.Signature_Size := Signature_Size ;
      Ctx.Signature := ( 0 => 16#67452301# ,
			 1 => 16#Efcdab89# ,
			 2 => 16#98badcfe# ,
			 3 => 16#10325476# ,
			 4 => 16#C3d2e1f0# ) ;
   end Initialize ;

   procedure Update( Ctx : in out Context_T ;
		     Data : Storage.Byte_Array_t ;
		     Data_Size : in Integer ) is 

     X : Storage.Long_Array_T( 0..15 ) ;
         for X'Address use Data'Address ;
      Aa , Bb , Cc , Dd , Ee : Interfaces.Unsigned_32 ;
      Aaa , Bbb , Ccc , Ddd , Eee : Interfaces.Unsigned_32 ;

   begin

      if Data_Size < Block_Size
      then
	 declare
	    Last_Block : Storage.Byte_Array_T(1..Block_Size) := Data ;
	    -- Add a single bit
	 begin
	    Last_block( Data_Size + 1 ) := 16#80# ;
	    Last_block( Data_size + 2 .. Block_Size ) := ( others => 0) ;
	    
	    if Data_Size + 2 > Block_Size - 8
	    then
	       Update( Ctx , Last_Block , Block_Size ) ;
	       Last_Block := (others => 0) ;
	    end if ;
	    Last_Block( Block_Size-7 .. Block_Size ) := Storage.Unpack( 8 * Ctx.Byte_Count ) ;                
	    Update(Ctx,Last_Block,Block_Size) ;
	 end ;
	 Ctx.Finalized := True ;
	 return ;
      end if ;

      Aa := Ctx.Signature(0) ;
      Bb := Ctx.Signature(1) ;
      Cc := Ctx.Signature(2) ;
      Dd := Ctx.Signature(3) ;
      Ee := Ctx.Signature(4) ;

      Aaa := Aa ;
      Bbb := Bb ;
      Ccc := Cc ;
      Ddd := Dd ;
      Eee := Ee ;

      -- Round 1
      Ff1(Aa,Bb,Cc,Dd,Ee,X(0), 11) ;
      Ff1(Ee,Aa,Bb,Cc,Dd,X(1), 14) ;
      Ff1(Dd,Ee,Aa,Bb,Cc,X(2), 15) ;
      Ff1(Cc,Dd,Ee,Aa,Bb,X(3), 12) ;
      Ff1(Bb,Cc,Dd,Ee,Aa,X(4),  5) ;
      Ff1(Aa,Bb,Cc,Dd,Ee,X(5),  8) ;
      Ff1(Ee,Aa,Bb,Cc,Dd,X(6),  7) ;
      Ff1(Dd,Ee,Aa,Bb,Cc,X(7),  9) ;
      Ff1( Cc,Dd,Ee,Aa,Bb,X(8),11) ;
      Ff1( Bb,Cc,Dd,Ee,Aa,X(9),13) ;
      Ff1(Aa,Bb,Cc,Dd,Ee,X(10),14) ;
      Ff1(Ee,Aa,Bb,Cc,Dd,X(11),15) ;
      Ff1(Dd,Ee,Aa,Bb,Cc,X(12), 6) ;
      Ff1(Cc,Dd,Ee,Aa,Bb,X(13), 7) ;
      Ff1(Bb,Cc,Dd,Ee,Aa,X(14), 9) ;
      Ff1(Aa,Bb,Cc,Dd,Ee,X(15), 8) ;

      -- Round 2 
      Ff2(Ee,Aa,Bb,Cc,Dd,X(7),  7) ;
      Ff2(Dd,Ee,Aa,Bb,Cc,X(4) , 6) ;
      Ff2(Cc,Dd,Ee,Aa,Bb,X(13), 8) ;
      Ff2(Bb,Cc,Dd,Ee,Aa,X(1), 13) ;
      Ff2(Aa,Bb,Cc,Dd,Ee,X(10),11) ;
      Ff2(Ee,Aa,Bb,Cc,Dd,X(6),  9) ;
      Ff2(Dd,Ee,Aa,Bb,Cc,X(15), 7) ;
      Ff2(Cc,Dd,Ee,Aa,Bb,X(3), 15) ;
      Ff2(Bb,Cc,Dd,Ee,Aa,X(12), 7) ;
      Ff2(Aa,Bb,Cc,Dd,Ee,X(0), 12) ;
      Ff2(Ee,Aa,Bb,Cc,Dd,X(9), 15) ;
      Ff2(Dd,Ee,Aa,Bb,Cc,X(5),  9) ;
      Ff2(Cc,Dd,Ee,Aa,Bb,X(2), 11) ;
      Ff2(Bb,Cc,Dd,Ee,Aa,X(14), 7) ;
      Ff2(Aa,Bb,Cc,Dd,Ee,X(11),13) ;
      Ff2(Ee,Aa,Bb,Cc,Dd,X(8), 12) ;

      -- Round 3
      Ff3(Dd,Ee,Aa,Bb,Cc,X(3) , 11) ;
      Ff3(Cc,Dd,Ee,Aa,Bb,X(10), 13) ;
      Ff3(Bb,Cc,Dd,Ee,Aa,X(14),6) ;
      Ff3(Aa,Bb,Cc,Dd,Ee,X(4),7) ;
      Ff3(Ee,Aa,Bb,Cc,Dd,X(9),14) ;
      Ff3(Dd,Ee,Aa,Bb,Cc,X(15), 9) ;
      Ff3(Cc,Dd,Ee,Aa,Bb,X(8) , 13) ;
      Ff3(Bb,Cc,Dd,Ee,Aa,X(1),15) ;
      Ff3(Aa,Bb,Cc,Dd,Ee,X(2) , 14) ;
      Ff3(Ee,Aa,Bb,Cc,Dd,X(7),8) ;
      Ff3(Dd,Ee,Aa,Bb,Cc,X(0),13) ;
      Ff3(Cc,Dd,Ee,Aa,Bb,X(6),6) ;
      Ff3(Bb,Cc,Dd,Ee,Aa,X(13),5) ;
      Ff3(Aa,Bb,Cc,Dd,Ee,X(11),12);
      Ff3(Ee,Aa,Bb,Cc,Dd,X(5),7) ;
      Ff3(Dd,Ee,Aa,Bb,Cc,X(12),5) ;

      -- Round 4
      Ff4(Cc,Dd,Ee,Aa,Bb,X(1),11) ;
      Ff4(Bb,Cc,Dd,Ee,Aa,X(9),12) ;
      Ff4(Aa,Bb,Cc,Dd,Ee,X(11),14) ;
      Ff4(Ee,Aa,Bb,Cc,Dd,X(10),15) ;
      Ff4(Dd,Ee,Aa,Bb,Cc,X(0),14) ;
      Ff4(Cc,Dd,Ee,Aa,Bb,X(8),15) ;
      Ff4(Bb,Cc,Dd,Ee,Aa,X(12),9) ;
      Ff4(Aa,Bb,Cc,Dd,Ee,X(4),8) ;
      Ff4(Ee,Aa,Bb,Cc,Dd,X(13),9) ;
      Ff4(Dd,Ee,Aa,Bb,Cc,X(3), 14) ;
      Ff4(Cc,Dd,Ee,Aa,Bb,X(7),5) ;
      Ff4(Bb,Cc,Dd,Ee,Aa,X(15),6) ;
      Ff4(Aa,Bb,Cc,Dd,Ee,X(14),8) ;
      Ff4(Ee,Aa,Bb,Cc,Dd,X(5),6) ;
      Ff4(Dd,Ee,Aa,Bb,Cc,X(6),5) ;
      Ff4(Cc,Dd,Ee,Aa,Bb,X(2),12) ;

      -- Round 5
      Ff5(Bb,Cc,Dd,Ee,Aa,X(4),9) ;
      Ff5(Aa,Bb,Cc,Dd,Ee,X(0),15) ;
      Ff5(Ee,Aa,Bb,Cc,Dd,X(5),5) ;
      Ff5(Dd,Ee,Aa,Bb,Cc,X(9),11) ;
      Ff5(Cc,Dd,Ee,Aa,Bb,X(7),6) ;
      Ff5(Bb,Cc,Dd,Ee,aa,X(12),8) ;
      Ff5(Aa,Bb,Cc,Dd,Ee,X(2),13) ;
      Ff5(Ee,Aa,Bb,Cc,Dd,X(10),12) ;
      Ff5(Dd,Ee,Aa,Bb,Cc,X(14),5) ;
      Ff5(Cc,Dd,Ee,Aa,Bb,X(1),12) ;
      Ff5(Bb,Cc,Dd,Ee,Aa,X(3),13) ;
      Ff5(Aa,Bb,Cc,Dd,Ee,X(8),14) ;
      Ff5(Ee,Aa,Bb,Cc,Dd,X(11),11);
      Ff5(Dd,Ee,Aa,Bb,Cc,X(6),8) ;
      Ff5(Cc,Dd,Ee,Aa,Bb,X(15),5) ;
      Ff5(Bb,Cc,Dd,Ee,Aa,X(13),6) ;

      ---------------------------------------------------------------------------

      -- Parallel Round 1
      Fff5(Aaa,Bbb,Ccc,Ddd,Eee,X(5),8) ;
      Fff5(Eee,Aaa,Bbb,Ccc,Ddd,X(14),9) ;
      Fff5(Ddd,Eee,Aaa,Bbb,Ccc,X(7),9) ;
      Fff5(Ccc,Ddd,Eee,Aaa,Bbb,X(0),11) ;
      Fff5(Bbb,Ccc,Ddd,Eee,Aaa,X(9),13) ;
      Fff5(Aaa,Bbb,Ccc,Ddd,Eee,X(2),15) ;
      Fff5(Eee,Aaa,Bbb,Ccc,Ddd,X(11),15);
      Fff5(Ddd,Eee,Aaa,Bbb,Ccc,X(4),5) ;
      Fff5(Ccc,Ddd,Eee,Aaa,Bbb,X(13),7) ;
      Fff5(Bbb,Ccc,Ddd,Eee,Aaa,X(6),7) ;
      Fff5(Aaa,Bbb,Ccc,Ddd,Eee,X(15),8) ;
      Fff5(Eee,Aaa,Bbb,Ccc,Ddd,X(8),11) ;
      Fff5(Ddd,Eee,Aaa,Bbb,Ccc,X(1),14) ;
      Fff5(Ccc,Ddd,Eee,Aaa,Bbb,X(10),14) ;
      Fff5(Bbb,Ccc,Ddd,Eee,Aaa,X(3),12) ;
      Fff5(Aaa,Bbb,Ccc,Ddd,Eee,X(12),6) ;

      -- Parallel Round 2
      Fff4(Eee,Aaa,Bbb,Ccc,Ddd,X(6),9) ;
      Fff4(Ddd,Eee,Aaa,Bbb,Ccc,X(11),13) ;
      Fff4(Ccc,Ddd,Eee,Aaa,Bbb,X(3),15) ;
      Fff4(Bbb,Ccc,Ddd,Eee,Aaa,X(7),7) ;
      Fff4(Aaa,Bbb,Ccc,Ddd,Eee,X(0),12) ;
      Fff4(Eee,Aaa,Bbb,Ccc,Ddd,X(13),8) ;
      Fff4(Ddd,Eee,Aaa,Bbb,Ccc,X(5),9) ;
      Fff4(Ccc,Ddd,Eee,Aaa,Bbb,X(10),11) ;
      Fff4(Bbb,Ccc,Ddd,Eee,Aaa,X(14),7) ;
      Fff4(Aaa,Bbb,Ccc,Ddd,Eee,X(15),7) ;
      Fff4(Eee,Aaa,Bbb,Ccc,Ddd,X(8),12) ;
      Fff4(Ddd,Eee,Aaa,Bbb,Ccc,X(12),7) ;
      Fff4(Ccc,Ddd,Eee,Aaa,Bbb,X(4),6) ;
      Fff4(Bbb,Ccc,Ddd,Eee,Aaa,X(9),15) ;
      Fff4(Aaa,Bbb,Ccc,Ddd,Eee,X(1),13) ;
      Fff4(Eee,Aaa,Bbb,Ccc,Ddd,X(2),11) ;

      -- Parallel Round 3
      Fff3(Ddd,Eee,Aaa,Bbb,Ccc,X(15),9) ;
      Fff3(Ccc,Ddd,Eee,Aaa,Bbb,X(5),7) ;
      Fff3(Bbb,Ccc,Ddd,Eee,Aaa,X(1),15) ;
      Fff3(Aaa,Bbb,Ccc,Ddd,Eee,X(3),11) ;
      Fff3(Eee,Aaa,Bbb,Ccc,Ddd,X(7),8) ;
      Fff3(Ddd,Eee,Aaa,Bbb,Ccc,X(14),6) ;
      Fff3(Ccc,Ddd,Eee,Aaa,Bbb,X(6),6) ;
      Fff3(Bbb,Ccc,Ddd,Eee,Aaa,X(9),14) ;
      Fff3(Aaa,Bbb,Ccc,Ddd,Eee,X(11),12) ;
      Fff3(Eee,Aaa,Bbb,Ccc,Ddd,X(8),13) ;
      Fff3(Ddd,Eee,Aaa,Bbb,Ccc,X(12),5) ;
      Fff3(Ccc,Ddd,Eee,Aaa,Bbb,X(2),14) ;
      Fff3(Bbb,Ccc,Ddd,Eee,Aaa,X(10),13) ;
      Fff3(Aaa,Bbb,Ccc,Ddd,Eee,X(0),13) ;
      Fff3(Eee,Aaa,Bbb,Ccc,Ddd,X(4),7) ;
      Fff3(Ddd,Eee,Aaa,Bbb,Ccc,X(13),5) ;

      -- Parallel Round 4
      Fff2(Ccc,Ddd,Eee,Aaa,Bbb,X(8),15) ;
      Fff2(Bbb,Ccc,Ddd,Eee,Aaa,X(6),5) ;
      Fff2(Aaa,Bbb,Ccc,Ddd,Eee,X(4),8) ;
      Fff2(eee,aaa,bbb,ccc,ddd,X(1),11) ;
      Fff2(Ddd,Eee,Aaa,Bbb,Ccc,X(3),14) ;
      Fff2(Ccc,Ddd,Eee,Aaa,Bbb,X(11),14) ;
      Fff2(Bbb,Ccc,Ddd,Eee,Aaa,X(15),6) ;
      Fff2(Aaa,Bbb,Ccc,Ddd,Eee,X(0),14) ;
      Fff2(Eee,Aaa,Bbb,Ccc,Ddd,X(5),6) ;
      Fff2(Ddd,Eee,Aaa,Bbb,Ccc,X(12),9) ;
      Fff2(Ccc,Ddd,Eee,Aaa,Bbb,X(2),12) ;
      Fff2(Bbb,Ccc,Ddd,Eee,Aaa,X(13),9) ;
      Fff2(Aaa,Bbb,Ccc,Ddd,Eee,X(9),12) ;
      Fff2(Eee,Aaa,Bbb,Ccc,Ddd,X(7),5) ;
      Fff2(Ddd,Eee,Aaa,Bbb,Ccc,X(10),15) ;
      Fff2(Ccc,Ddd,Eee,Aaa,Bbb,X(14),8) ;

      -- Parallel Round 5
      Fff1(Bbb,Ccc,Ddd,Eee,Aaa,X(12),8) ;
      Fff1(Aaa,Bbb,Ccc,Ddd,Eee,X(15),5) ;
      Fff1(Eee,Aaa,Bbb,Ccc,Ddd,X(10),12) ;
      Fff1(Ddd,Eee,Aaa,Bbb,Ccc,X(4),9) ;
      Fff1(Ccc,Ddd,Eee,Aaa,Bbb,X(1),12) ;
      Fff1(Bbb,Ccc,Ddd,Eee,Aaa,X(5),5) ;
      Fff1(Aaa,Bbb,Ccc,Ddd,Eee,X(8),14) ;
      Fff1(Eee,Aaa,Bbb,Ccc,Ddd,X(7),6) ;
      Fff1(Ddd,Eee,Aaa,Bbb,Ccc,X(6),8) ;
      Fff1(Ccc,Ddd,Eee,Aaa,Bbb,X(2),13) ;
      Fff1(Bbb,Ccc,Ddd,Eee,Aaa,X(13),6) ;
      Fff1(Aaa,Bbb,Ccc,Ddd,Eee,X(14),5) ;
      Fff1(Eee,Aaa,Bbb,Ccc,Ddd,X(0),15) ;
      Fff1(Ddd,Eee,Aaa,Bbb,Ccc,X(3),13) ;
      Fff1(Ccc,Ddd,Eee,Aaa,Bbb,X(9),11) ;
      Fff1(Bbb,Ccc,Ddd,Eee,Aaa,X(11),11) ;

      -- Final - Combine results
      Ddd := Ddd + Cc + Ctx.Signature(1) ;
      Ctx.Signature(1) := Ctx.Signature(2) + Dd + Eee ;
      Ctx.Signature(2) := Ctx.Signature(3) + Ee + Aaa ;
      Ctx.Signature(3) := Ctx.Signature(4) + Aa + Bbb ;
      Ctx.Signature(4) := Ctx.Signature(0) + Bb + Ccc ;
      Ctx.Signature(0) := Ddd ;

   end Update ;

   function  OneWayHash( CTX : in CONTEXT_T ) return Signature_T is
      Bit_Count : Interfaces.Unsigned_64 := 8 * Ctx.Byte_Count ;
      Signature : Storage.Byte_Array_T(1..Signature_Size) ;
   begin
      for I in Ctx.Signature'Range
      loop
	 Signature( 4 * I + 1 .. 4 * I + 4 ) :=
	   Storage.Unpack( Ctx.Signature(I) ) ;
      end loop ;
      return Signature_T( Signature ) ;
   end OneWayHash ;
end Encrypt.Ripemd160 ;
