
with Text_Io ;

package body Encrypt.Haval is
   use Interfaces ;
   
   procedure Initialize( Ctx : out Context_T ) is
   begin
      Initialize( Ctx , Rounds_T'First , Hashlength_T'Last ) ;
   end Initialize ;
   
   procedure Initialize( Ctx : out Context_T ;
         Rounds : in Rounds_T ;
         Hashlength : in Hashlength_T ) is                         
   begin
      Ctx.Block_Size := Block_Size ;
      Ctx.Byte_Count  := 0 ;
      Ctx.Rounds := Rounds ;
      
      case Hashlength is
         when H128 => Ctx.Signature_Size := 128/8 ;
         when H160 => Ctx.Signature_Size := 160/8 ;
         when H192 => Ctx.Signature_Size := 192/8 ;
         when H224 => Ctx.Signature_Size := 224/8 ;     
         when H256 => Ctx.Signature_Size := 256/8 ;        
      end case ;
      
      Ctx.Signature :=  ( 16#243F_6A88# ,
                          16#85A3_08D3# ,
			  16#1319_8A2E# ,
			  16#0370_7344# ,
                          16#A409_3822# ,
			  16#299F_31D0# ,
			  16#082E_FA98# ,
                          16#EC4E_6C89# ) ;
 
         Ctx.Initialized := True ;

      end Initialize ;
      
      Contextnow : Context_T ;
     
      function Ror (X : Interfaces.Unsigned_32 ; N : Natural )
         return Interfaces.Unsigned_32 is
      begin
         return (Interfaces.Shift_Right(X , N)  or
         Interfaces.Shift_Left(X , 32-N) ) ;
      end Ror ;
      pragma Inline(Ror) ;

      function F_1(X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32)
         return Interfaces.Unsigned_32 is
      begin       
         return (x1 and (x0 xor x4)) xor (x2 and x5) xor
                 ((x3 and x6) xor x0) ;
      end F_1 ;
      pragma Inline( F_1 ) ;
     function F_2(X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32 )
         return Interfaces.Unsigned_32 is
      begin
         return
           (x2 and ( (x1 and not(x3)) xor (x4 and x5) xor x6 xor x0)) xor 
            (x4 and (x1 xor x5)) xor (x3 and x5) xor x0 ;

--#define f_2(x6, x5, x4, x3, x2, x1, x0)                         \
--           ((x2) & ((x1) & ~(x3) ^ (x4) & (x5) ^ (x6) ^ (x0)) ^ \
--            (x4) & ((x1) ^ (x5)) ^ (x3) & (x5) ^ (x0))

      end F_2 ;      

      pragma Inline(F_2) ;
      
      function F_3 (X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32)
         return Interfaces.Unsigned_32 is
      begin
         return  (x3 and ((x1 and x2) xor x6 xor x0)) xor 
                       (x1 and x4) xor (x2 and x5) xor x0 ;
      end F_3 ;
      pragma Inline(F_3) ;
      function F_4(X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32 )
         return Interfaces.Unsigned_32 is
      begin
         return 
           (x4 and ( (x5 and not(x2)) xor (x3 and not(x6)) xor x1 xor x6 xor x0)) xor
            ((x3 and ((x1 and x2) xor x5 xor x6)) xor
            (x2 and x6) xor x0) ;


--#define f_4(x6, x5, x4, x3, x2, x1, x0)                                 \
--           ((x4) & ((x5) & ~(x2) ^ (x3) & ~(x6) ^ (x1) ^ (x6) ^ (x0)) ^ \
--            (x3) & ((x1) & (x2) ^ (x5) ^ (x6)) ^                        \
--            (x2) & (x6) ^ (x0))
      end F_4 ;
      

      pragma Inline(F_4) ;
      
      function F_5 (X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32)
         return Interfaces.Unsigned_32 is
      begin
         return (x0 and ((x1 and x2 and x3) xor not(x5))) xor
		 (x1 and x4) xor (x2 and x5) xor (x3 and x6) ;

      end F_5 ;
      pragma Inline(F_5) ;

      function Fphi_1(X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32 ) 
         return Interfaces.Unsigned_32 is
      begin
         case Contextnow.Rounds is
            when 3 => return F_1(X1, X0, X3, X5, X6, X2, X4) ;
            when 4 => return F_1(X2, X6, X1, X4, X5, X3, X0) ;
            when 5 => return F_1(X3, X4, X1, X0, X5, X2, X6) ;
         end case ;
      end Fphi_1 ;

      function Fphi_2(X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32 )  
         return Interfaces.Unsigned_32 is
      begin
         case Contextnow.Rounds is
            when 3 => return F_2(X4, X2, X1, X0, X5, X3, X6) ;
            when 4 => return F_2(X3, X5, X2, X0, X1, X6, X4) ;
            when 5 => return F_2(X6, X2, X1, X0, X3, X4, X5) ;
         end case ;
      end Fphi_2 ;

      function Fphi_3(X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32 )  
         return Interfaces.Unsigned_32 is
      begin
         case Contextnow.Rounds is
            when 3 => return F_3(X6, X1, X2, X3, X4, X5, X0) ;
            when 4 => return F_3(X1, X4, X3, X6, X0, X2, X5) ;
            when 5 => return F_3(X2, X6, X0, X4, X3, X1, X5) ;
         end case ;
      end Fphi_3 ;
      
      function Fphi_4(X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32 )  
         return Interfaces.Unsigned_32 is
      begin
         case Contextnow.Rounds is
            when 3 => raise Program_Error ;
            when 4 => return F_4(X6, X4, X0, X5, X2, X1, X3) ;
            when 5 => return F_4(X1, X5, X3, X2, X0, X4, X6) ;
         end case ;
      end Fphi_4 ;
      
      function Fphi_5(X6, X5, X4, X3, X2, X1, X0 : Interfaces.Unsigned_32 )  
         return Interfaces.Unsigned_32 is
      begin
         case Contextnow.Rounds is
            when 3 | 4 => raise Program_Error ;
            when 5 => return F_5(X2, X5, X0, X6, X4, X3, X1) ;
         end case ;
      end Fphi_5 ;

      procedure Ff_1(X7, X6, X5, X4, X3, X2, X1, X0 : in out Interfaces.Unsigned_32 ;
         W : Interfaces.Unsigned_32 ) is
         Temp : Interfaces.Unsigned_32 ;
      begin 
         Temp := Fphi_1(X6, X5, X4, X3, X2, X1, X0) ;
         X7 := Ror(Temp, 7) + Ror(X7,11) + W ; 
      end Ff_1 ;

      procedure Ff_2(X7, X6, X5, X4, X3, X2, X1, X0 : in out Interfaces.Unsigned_32 ;
         W : in Interfaces.Unsigned_32 ;
         C : in Interfaces.Unsigned_32 ) is
         Temp : Interfaces.Unsigned_32 ;

      begin
         Temp := Fphi_2(X6, X5, X4, X3, X2, X1, X0);
         X7 := Ror(Temp, 7) + Ror(X7 , 11) + W + C ;  
      end Ff_2 ;

      procedure Ff_3(X7, X6, X5, X4, X3, X2, X1, X0 : in out Interfaces.Unsigned_32 ;
         W : in Interfaces.Unsigned_32 ;
         C : in Interfaces.Unsigned_32 ) is

         Temp : Interfaces.Unsigned_32 ;

      begin                     
         Temp := Fphi_3(X6, X5, X4, X3, X2, X1, X0) ; 
         X7 := Ror(Temp, 7) + Ror(X7, 11) + W + C ;
      end Ff_3 ;

      procedure Ff_4(X7, X6, X5, X4, X3, X2, X1, X0 : in out Interfaces.Unsigned_32 ;
         W : in Interfaces.Unsigned_32 ;
         C : in Interfaces.Unsigned_32 ) is
         Temp : Interfaces.Unsigned_32 ;

      begin
         Temp := Fphi_4(X6, X5, X4, X3, X2, X1, X0) ;
         X7 := Ror(Temp, 7) + Ror(X7 , 11) + W + C ;
      end Ff_4 ;

      procedure Ff_5(X7, X6, X5, X4, X3, X2, X1, X0 : in out Interfaces.Unsigned_32 ;
         W : in Interfaces.Unsigned_32 ;
         C : in Interfaces.Unsigned_32 ) is
         Temp : Interfaces.Unsigned_32 ;

      begin
         Temp := Fphi_5(X6, X5, X4, X3, X2, X1, X0);
         X7 := Ror(Temp, 7) + Ror(X7 , 11) + W + C ; 
      end Ff_5 ;

      type Fptlen_T is range 0..3 ;
      for Fptlen_T'Size use 2 ;
      type Passes_T is range 0..7 ;
      for Passes_T'Size use 3 ;
      type Version_T is range 0..7 ;
      for Version_T'Size use 3 ;
      type Fingerprint_Byte_T is      
      record
         Version : Version_T ;
         Passes : Passes_T ;
         Fptlen : Fptlen_T ;
      end record ;
      for Fingerprint_Byte_T use
	 record
	    Version at 0 range 0..2 ;
	    Passes at 0 range 3..5 ;
	    Fptlen at 0 range 6..7 ;
	 end record ;
      pragma Pack( Fingerprint_Byte_T ) ;
      for Fingerprint_Byte_T'Size use 8 ;
      
      procedure Update( Ctx : in out Context_T ;
         Data : Storage.Byte_Array_T ;
         Data_Size : in Integer ) is
         T : Storage.Long_Array_T( 0..7 ) := Ctx.Signature ;       
         Datal : Storage.Long_Array_T(0..Block_Size/4-1) ;
         for Datal'Address use Data'Address ;
      begin

         if Data_Size < Ctx.Block_Size
            then
            declare
               Last_Block : Storage.Byte_Array_T(1..Block_Size) := Data ;
               Fpbyte : Fingerprint_Byte_T ;
               Fpbyte_B : Interfaces.Unsigned_8 ;
               for Fpbyte_B'Address use Fpbyte'Address ;
            begin
               Last_Block(Data_Size+1) := 16#80# ;               
	       Last_Block(Data_Size+2..Block_Size) := ( others => 0 ) ;
               if Data_Size+2 > Block_Size - 10
	       then                                  
                  Update( Ctx , Last_Block , Block_Size ) ;
                  Last_Block := (others => 0) ;
               end if ;

--               Fpbyte.Fptlen :=  Fptlen_T(Ctx.Signature_Size*8 mod 3) ;
               Fpbyte.Fptlen := 0 ;
               Fpbyte.Passes := Passes_T(Ctx.Rounds mod 7) ;
               Fpbyte.Version := Version_T(Version mod 7) ;
                
               Last_Block( Block_Size-7 .. Block_Size ) := Storage.Unpack( 8 * Ctx.Byte_Count ) ;
                
               Last_Block( Block_Size-8) := 
		         Interfaces.Unsigned_8( Ctx.Signature_Size * 8 / 4 ) ;
               Last_Block( Block_Size-9) := Interfaces.Unsigned_8( Fpbyte_B ) ;
               Update(Ctx,Last_Block,Block_Size) ;

            end ;
            Ctx.Finalized := True ;
            return ;
         end if ;      
         Contextnow := Ctx ;
         -- /* Pass 1 */
  
         Ff_1(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(0));
         Ff_1(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 1));
         Ff_1(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal( 2));
         Ff_1(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal( 3));
         Ff_1(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal( 4));
         Ff_1(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal( 5));
         Ff_1(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal( 6));
         Ff_1(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal( 7));

         Ff_1(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal( 8));
         Ff_1(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 9));
         Ff_1(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(10));
         Ff_1(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(11));
         Ff_1(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(12));
         Ff_1(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(13));
         Ff_1(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(14));
         Ff_1(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(15));

         Ff_1(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(16));
         Ff_1(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(17));
         Ff_1(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(18));
         Ff_1(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(19));
         Ff_1(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(20));
         Ff_1(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(21));
         Ff_1(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(22));
         Ff_1(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(23));

         Ff_1(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(24));
         Ff_1(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(25));
         Ff_1(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(26));
         Ff_1(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(27));
         Ff_1(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(28));
         Ff_1(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(29));
         Ff_1(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(30));
         Ff_1(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(31));

         -- /* Pass 2 */
         Ff_2(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal( 5), 16#452821E6# ) ;
         Ff_2(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(14), 16#38D01377# ) ;
         Ff_2(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(26), 16#BE5466CF# ) ;
         Ff_2(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(18), 16#34E90C6C# ) ;
         Ff_2(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(11), 16#C0AC29B7# ) ;
         Ff_2(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(28), 16#C97C50DD# ) ;
         Ff_2(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal( 7), 16#3F84D5B5# ) ;
         Ff_2(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(16), 16#B5470917# ) ;

         Ff_2(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(0 ), 16#9216D5D9# ) ;
         Ff_2(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(23), 16#8979FB1B# ) ;
         Ff_2(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(20), 16#D1310BA6# ) ;
         Ff_2(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(22), 16#98DFB5AC# ) ;
         Ff_2(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal( 1), 16#2FFD72DB# ) ;
         Ff_2(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(10), 16#D01ADFB7# ) ;
         Ff_2(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal( 4), 16#B8E1AFED# ) ;
         Ff_2(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal( 8), 16#6A267E96# ) ;

         Ff_2(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(30), 16#BA7C9045# ) ;
         Ff_2(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 3), 16#F12C7F99# ) ;
         Ff_2(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(21), 16#24A19947# ) ;
         Ff_2(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal( 9), 16#B3916CF7# ) ;
         Ff_2(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(17), 16#0801F2E2# ) ;
         Ff_2(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(24), 16#858EFC16# ) ;
         Ff_2(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(29), 16#636920D8# ) ;
         Ff_2(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal( 6), 16#71574E69# ) ;

         Ff_2(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(19), 16#A458FEA3# ) ;
         Ff_2(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(12), 16#F4933D7E# ) ;
         Ff_2(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(15), 16#0D95748F# ) ;
         Ff_2(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(13), 16#728EB658# ) ;
         Ff_2(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal( 2), 16#718BCD58# ) ;
         Ff_2(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(25), 16#82154AEE# ) ;
         Ff_2(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(31), 16#7B54A41D# ) ;
         Ff_2(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(27), 16#C25A59B5# ) ;

         -- /* Pass 3 */
         Ff_3(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(19), 16#9C30D539# ) ;
         Ff_3(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 9), 16#2AF26013# ) ;
         Ff_3(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal( 4), 16#C5D1B023# ) ;
         Ff_3(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(20), 16#286085F0# ) ;
         Ff_3(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(28), 16#CA417918# ) ;
         Ff_3(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(17), 16#B8DB38EF# ) ;
         Ff_3(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal( 8), 16#8E79DCB0# ) ;
         Ff_3(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(22), 16#603A180E# ) ;

         Ff_3(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(29), 16#6C9E0E8B# ) ;
         Ff_3(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(14), 16#B01E8A3E# ) ;
         Ff_3(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(25), 16#D71577C1# ) ;
         Ff_3(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(12), 16#BD314B27# ) ;
         Ff_3(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(24), 16#78AF2FDA# ) ;
         Ff_3(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(30), 16#55605C60# ) ;
         Ff_3(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(16), 16#E65525F3# ) ;
         Ff_3(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(26), 16#AA55AB94# ) ;

         Ff_3(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(31), 16#57489862# ) ;
         Ff_3(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(15), 16#63E81440# ) ;
         Ff_3(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal( 7), 16#55CA396A# ) ;
         Ff_3(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal( 3), 16#2AAB10B6# ) ;
         Ff_3(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal( 1), 16#B4CC5C34# ) ;
         Ff_3(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(0 ), 16#1141E8CE# ) ;
         Ff_3(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(18), 16#A15486AF# ) ;
         Ff_3(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(27), 16#7C72E993# ) ;

         Ff_3(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(13), 16#B3EE1411# ) ;
         Ff_3(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 6), 16#636FBC2A# ) ;
         Ff_3(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(21), 16#2BA9C55D# ) ;
         Ff_3(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(10), 16#741831F6# ) ;
         Ff_3(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(23), 16#CE5C3E16# ) ;
         Ff_3(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(11), 16#9B87931E# ) ;
         Ff_3(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal( 5), 16#AFD6BA33# ) ;
         Ff_3(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal( 2), 16#6C24CF5C# ) ;

         if Ctx.Rounds >= 4
            then
            -- /* Pass 4. executed only when PASS =4 or 5 */
            Ff_4(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(24), 16#7A325381# ) ;
            Ff_4(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 4), 16#28958677# ) ;
            Ff_4(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(0 ), 16#3B8F4898# ) ;
            Ff_4(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(14), 16#6B4BB9AF# ) ;
            Ff_4(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal( 2), 16#C4BFE81B# ) ;
            Ff_4(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal( 7), 16#66282193# ) ;
            Ff_4(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(28), 16#61D809CC# ) ;
            Ff_4(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(23), 16#FB21A991# ) ;

            Ff_4(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(26), 16#487CAC60# ) ;
            Ff_4(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 6), 16#5DEC8032# ) ;
            Ff_4(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(30), 16#EF845D5D# ) ;
            Ff_4(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(20), 16#E98575B1# ) ;
            Ff_4(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(18), 16#DC262302# ) ;
            Ff_4(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(25), 16#EB651B88# ) ;
            Ff_4(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(19), 16#23893E81# ) ;
            Ff_4(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal( 3), 16#D396ACC5# ) ;
 
            Ff_4(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(22), 16#0F6D6FF3# ) ;
            Ff_4(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(11), 16#83F44239# ) ;
            Ff_4(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(31), 16#2E0B4482# ) ;
            Ff_4(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(21), 16#A4842004# ) ;
            Ff_4(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal( 8), 16#69C8F04A# ) ;
            Ff_4(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(27), 16#9E1F9B5E# ) ;
            Ff_4(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(12), 16#21C66842# ) ;
            Ff_4(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal( 9), 16#F6E96C9A# ) ;

            Ff_4(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal( 1), 16#670C9C61# ) ;
            Ff_4(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(29), 16#ABD388F0# ) ;
            Ff_4(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal( 5), 16#6A51A0D2# ) ;
            Ff_4(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(15), 16#D8542F68# ) ;
            Ff_4(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(17), 16#960FA728# ) ;
            Ff_4(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(10), 16#AB5133A3# ) ;
            Ff_4(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(16), 16#6EEF0B6C# ) ;
            Ff_4(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(13), 16#137A3BE4# ) ;
      
         end if ;
   

         if Ctx.Rounds = 5
            then
            -- /* Pass 5. executed only when PASS = 5 */
            Ff_5(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(27), 16#BA3BF050# ) ;
            Ff_5(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 3), 16#7EFB2A98# ) ;
            Ff_5(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(21), 16#A1F1651D# ) ;
            Ff_5(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(26), 16#39AF0176# ) ;
            Ff_5(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(17), 16#66CA593E# ) ;
            Ff_5(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal(11), 16#82430E88# ) ;
            Ff_5(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(20), 16#8CEE8619# ) ;
            Ff_5(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(29), 16#456F9FB4# ) ;

            Ff_5(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal(19), 16#7D84A5C3# ) ;
            Ff_5(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(0 ), 16#3B8B5EBE# ) ;
            Ff_5(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(12), 16#E06F75D8# ) ;
            Ff_5(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal( 7), 16#85C12073# ) ;
            Ff_5(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(13), 16#401A449F# ) ;
            Ff_5(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal( 8), 16#56C16AA6# ) ;
            Ff_5(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(31), 16#4ED3AA62# ) ;
            Ff_5(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(10), 16#363F7706# ) ;

            Ff_5(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal( 5), 16#1BFEDF72# ) ;
            Ff_5(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal( 9), 16#429B023D# ) ;
            Ff_5(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(14), 16#37D0D724# ) ;
            Ff_5(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(30), 16#D00A1248# ) ;
            Ff_5(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal(18), 16#DB0FEAD3# ) ;
            Ff_5(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal( 6), 16#49F1C09B# ) ;
            Ff_5(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(28), 16#075372C9# ) ;
            Ff_5(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(24), 16#80991B7B# ) ;

            Ff_5(T(7), T(6), T(5), T(4), T(3), T(2), T(1), T(0), Datal( 2), 16#25D479D8# ) ;
            Ff_5(T(6), T(5), T(4), T(3), T(2), T(1), T(0), T(7), Datal(23), 16#F6E8DEF7# ) ;
            Ff_5(T(5), T(4), T(3), T(2), T(1), T(0), T(7), T(6), Datal(16), 16#E3FE501A# ) ;
            Ff_5(T(4), T(3), T(2), T(1), T(0), T(7), T(6), T(5), Datal(22), 16#B6794C3B# ) ;
            Ff_5(T(3), T(2), T(1), T(0), T(7), T(6), T(5), T(4), Datal( 4), 16#976CE0BD# ) ;
            Ff_5(T(2), T(1), T(0), T(7), T(6), T(5), T(4), T(3), Datal( 1), 16#04C006BA# ) ;
            Ff_5(T(1), T(0), T(7), T(6), T(5), T(4), T(3), T(2), Datal(25), 16#C1A94FB6# ) ;
            Ff_5(T(0), T(7), T(6), T(5), T(4), T(3), T(2), T(1), Datal(15), 16#409F60C4# ) ;
         end if ;

         Ctx.Signature(0) := Ctx.Signature(0) + T(0) ;
         Ctx.Signature(1) := Ctx.Signature(1) + T(1) ;
         Ctx.Signature(2) := Ctx.Signature(2) + T(2) ;
         Ctx.Signature(3) := Ctx.Signature(3) + T(3) ;
         Ctx.Signature(4) := Ctx.Signature(4) + T(4) ;
         Ctx.Signature(5) := Ctx.Signature(5) + T(5) ;
         Ctx.Signature(6) := Ctx.Signature(6) + T(6) ;
         Ctx.Signature(7) := Ctx.Signature(7) + T(7) ;
       
      end Update ;

      function Onewayhash( Ctx : in Context_T ) return Signature_T is

         Temp : Interfaces.Unsigned_32 ;
         Csig : Storage.Long_Array_T( 0..7 ) := Ctx.Signature ;
         Sig : Signature_T(1..Ctx.Signature_Size) ;
	       for Sig'Address use Csig'Address ;
      begin 

         case Ctx.Signature_Size is       
            when 128/8 =>
               Temp := (Csig(7) and 16#000000FF#) or 
               (Csig(6) and 16#FF000000#) or 
               (Csig(5) and 16#00FF0000#) or
               (Csig(4) and 16#0000FF00#);
               Csig(0) := Csig(0) + Ror(Temp ,  8);

               Temp := (Csig(7) and 16#0000FF00#) or
               (Csig(6) and 16#000000FF#) or 
               (Csig(5) and 16#FF000000#) or 
               (Csig(4) and 16#00FF0000#) ;
               Csig(1) := Csig(1) + Ror(Temp, 16);

               Temp  := (Csig(7) and 16#00FF0000#) or 
               (Csig(6) and 16#0000FF00#) or 
               (Csig(5) and 16#000000FF#) or 
               (Csig(4) and 16#FF000000#);
               Csig(2) := Csig(2) + Ror(Temp, 24);

               Temp := (Csig(7) and 16#FF000000#) or
               (Csig(6) and 16#00FF0000#) or
               (Csig(5) and 16#0000FF00#) or
               (Csig(4) and 16#000000FF#);
               Csig(3) := Csig(3) + Temp;
                          
            when 160/8 =>
               Temp := (Csig(7) and  16#3F#) or 
               (Csig(6) and Interfaces.Shift_Left(16#7F# , 25)) or  
               (Csig(5) and Interfaces.Shift_Left(16#3F# , 19));
               Csig(0) := Csig(0) + Ror(Temp, 19);

               Temp := (Csig(7) and Interfaces.Shift_Left(16#3F# , 6)) or 
               (Csig(6) and  16#3F#) or  
               (Csig(5) and Interfaces.Shift_Left(16#7F# , 25));
               Csig(1) := Csig(1) + Ror(Temp, 25);

               Temp := (Csig(7) and Interfaces.Shift_Left(16#7F# , 12)) or 
               (Csig(6) and Interfaces.Shift_Left(16#3F# ,  6)) or  
               (Csig(5) and  16#3F# );
               Csig(2) := Csig(2) + Temp;

               Temp := (Csig(7) and Interfaces.Shift_Left(16#3F# , 19)) or 
               (Csig(6) and Interfaces.Shift_Left(16#7F# , 12)) or  
               (Csig(5) and Interfaces.Shift_Left(16#3F# ,  6));
               Csig(3) := Csig(3) + Interfaces.Shift_Right(Temp,6) ; 

               Temp := (Csig(7) and Interfaces.Shift_Left(16#7F# , 25)) or 
               (Csig(6) and Interfaces.Shift_Left(16#3F# , 19)) or  
               (Csig(5) and Interfaces.Shift_Left(16#7F# , 12));
               Csig(4) := Csig(4) + Interfaces.Shift_Right(Temp , 12) ;

            when 192/8 =>
	       temp := (csig(7) and 16#1F#) or 
		       (csig(6) and (Interfaces.Shift_Left(16#3F# , 26)));
	       csig(0) := Csig(0) + Ror(temp, 26) ;

	       temp :=  (csig(7) and Interfaces.Shift_Left(16#1F# , 5)) or 
		        (csig(6) and 16#1F#) ;
	       csig(1) := Csig(1) + Temp ;

	       temp := (csig(7) and Interfaces.Shift_Left(16#3F# , 10)) or 
		       (csig(6) and (Interfaces.Shift_Left(16#1F# , 5))) ;
	       csig(2) := Csig(2) + Interfaces.Shift_Right(temp , 5) ;

	       temp := (csig(7) and (Interfaces.Shift_Left(16#1F# , 16))) or 
		      (csig(6) and (Interfaces.Shift_Left(16#3F# , 10)));
	       csig(3) := Csig(3) + Interfaces.Shift_Right(Temp , 10) ;

	       temp := (csig(7) and (Interfaces.Shift_Left(16#1F# , 21))) or 
		      (csig(6) and (Interfaces.Shift_Left(16#1F# , 16)));
	       csig(4) := Csig(4) + Interfaces.Shift_Right(temp , 16) ;

	       temp := (csig(7) and (Interfaces.Shift_Left(16#3F# , 26))) or 
		       (csig(6) and (Interfaces.Shift_Left(16#1F# , 21))) ;
	       csig(5) := Csig(5) + Interfaces.Shift_Right(temp , 21) ;

            when 224/8 => 
	       csig(0) := Csig(0) + (Interfaces.Shift_Right(csig(7) , 27) and 16#1F#);
	       csig(1) := Csig(1) + (Interfaces.Shift_Right(csig(7) , 22) and 16#1F#);
	       csig(2) := Csig(2) + (Interfaces.Shift_Right(csig(7) , 18) and 16#0F#);
	       csig(3) := Csig(3) + (Interfaces.Shift_Right(csig(7) , 13) and 16#1F#);
	       csig(4) := Csig(4) + (Interfaces.Shift_Right(csig(7) ,  9) and 16#0F#);
	       csig(5) := Csig(5) + (Interfaces.Shift_Right(csig(7) ,  4) and 16#1F#);
	       csig(6) := Csig(6) + (csig(7) and 16#0F#) ;
            when 256/8 => null ;
            when others => raise Program_Error ;
         end case ;         
         return Sig ;
      end Onewayhash ;

   end Encrypt.Haval ;


